/*!
 * Bowser - a browser detector
 * https://github.com/ded/bowser
 * MIT License | (c) Dustin Diaz 2015
 */

!function (name, definition) {
  if (typeof module != 'undefined' && module.exports) module.exports = definition()
  else if (typeof define == 'function' && define.amd) define(name, definition)
  else this[name] = definition()
}('bowser', function () {
  /**
    * See useragents.js for examples of navigator.userAgent
    */

  var t = true

  function detect(ua) {

    function getFirstMatch(regex) {
      var match = ua.match(regex);
      return (match && match.length > 1 && match[1]) || '';
    }

    function getSecondMatch(regex) {
      var match = ua.match(regex);
      return (match && match.length > 1 && match[2]) || '';
    }

    var iosdevice = getFirstMatch(/(ipod|iphone|ipad)/i).toLowerCase()
      , likeAndroid = /like android/i.test(ua)
      , android = !likeAndroid && /android/i.test(ua)
      , nexusMobile = /nexus\s*[0-6]\s*/i.test(ua)
      , nexusTablet = !nexusMobile && /nexus\s*[0-9]+/i.test(ua)
      , chromeos = /CrOS/.test(ua)
      , silk = /silk/i.test(ua)
      , sailfish = /sailfish/i.test(ua)
      , tizen = /tizen/i.test(ua)
      , webos = /(web|hpw)os/i.test(ua)
      , windowsphone = /windows phone/i.test(ua)
      , windows = !windowsphone && /windows/i.test(ua)
      , mac = !iosdevice && !silk && /macintosh/i.test(ua)
      , linux = !android && !sailfish && !tizen && !webos && /linux/i.test(ua)
      , edgeVersion = getFirstMatch(/edge\/(\d+(\.\d+)?)/i)
      , versionIdentifier = getFirstMatch(/version\/(\d+(\.\d+)?)/i)
      , tablet = /tablet/i.test(ua)
      , mobile = !tablet && /[^-]mobi/i.test(ua)
      , xbox = /xbox/i.test(ua)
      , result

    if (/opera|opr|opios/i.test(ua)) {
      result = {
        name: 'Opera'
      , opera: t
      , version: versionIdentifier || getFirstMatch(/(?:opera|opr|opios)[\s\/](\d+(\.\d+)?)/i)
      }
    }
    else if (/coast/i.test(ua)) {
      result = {
        name: 'Opera Coast'
        , coast: t
        , version: versionIdentifier || getFirstMatch(/(?:coast)[\s\/](\d+(\.\d+)?)/i)
      }
    }
    else if (/yabrowser/i.test(ua)) {
      result = {
        name: 'Yandex Browser'
      , yandexbrowser: t
      , version: versionIdentifier || getFirstMatch(/(?:yabrowser)[\s\/](\d+(\.\d+)?)/i)
      }
    }
    else if (/ucbrowser/i.test(ua)) {
      result = {
          name: 'UC Browser'
        , ucbrowser: t
        , version: getFirstMatch(/(?:ucbrowser)[\s\/](\d+(?:\.\d+)+)/i)
      }
    }
    else if (/mxios/i.test(ua)) {
      result = {
        name: 'Maxthon'
        , maxthon: t
        , version: getFirstMatch(/(?:mxios)[\s\/](\d+(?:\.\d+)+)/i)
      }
    }
    else if (/epiphany/i.test(ua)) {
      result = {
        name: 'Epiphany'
        , epiphany: t
        , version: getFirstMatch(/(?:epiphany)[\s\/](\d+(?:\.\d+)+)/i)
      }
    }
    else if (/puffin/i.test(ua)) {
      result = {
        name: 'Puffin'
        , puffin: t
        , version: getFirstMatch(/(?:puffin)[\s\/](\d+(?:\.\d+)?)/i)
      }
    }
    else if (/sleipnir/i.test(ua)) {
      result = {
        name: 'Sleipnir'
        , sleipnir: t
        , version: getFirstMatch(/(?:sleipnir)[\s\/](\d+(?:\.\d+)+)/i)
      }
    }
    else if (/k-meleon/i.test(ua)) {
      result = {
        name: 'K-Meleon'
        , kMeleon: t
        , version: getFirstMatch(/(?:k-meleon)[\s\/](\d+(?:\.\d+)+)/i)
      }
    }
    else if (windowsphone) {
      result = {
        name: 'Windows Phone'
      , windowsphone: t
      }
      if (edgeVersion) {
        result.msedge = t
        result.version = edgeVersion
      }
      else {
        result.msie = t
        result.version = getFirstMatch(/iemobile\/(\d+(\.\d+)?)/i)
      }
    }
    else if (/msie|trident/i.test(ua)) {
      result = {
        name: 'Internet Explorer'
      , msie: t
      , version: getFirstMatch(/(?:msie |rv:)(\d+(\.\d+)?)/i)
      }
    } else if (chromeos) {
      result = {
        name: 'Chrome'
      , chromeos: t
      , chromeBook: t
      , chrome: t
      , version: getFirstMatch(/(?:chrome|crios|crmo)\/(\d+(\.\d+)?)/i)
      }
    } else if (/chrome.+? edge/i.test(ua)) {
      result = {
        name: 'Microsoft Edge'
      , msedge: t
      , version: edgeVersion
      }
    }
    else if (/vivaldi/i.test(ua)) {
      result = {
        name: 'Vivaldi'
        , vivaldi: t
        , version: getFirstMatch(/vivaldi\/(\d+(\.\d+)?)/i) || versionIdentifier
      }
    }
    else if (sailfish) {
      result = {
        name: 'Sailfish'
      , sailfish: t
      , version: getFirstMatch(/sailfish\s?browser\/(\d+(\.\d+)?)/i)
      }
    }
    else if (/seamonkey\//i.test(ua)) {
      result = {
        name: 'SeaMonkey'
      , seamonkey: t
      , version: getFirstMatch(/seamonkey\/(\d+(\.\d+)?)/i)
      }
    }
    else if (/firefox|iceweasel|fxios/i.test(ua)) {
      result = {
        name: 'Firefox'
      , firefox: t
      , version: getFirstMatch(/(?:firefox|iceweasel|fxios)[ \/](\d+(\.\d+)?)/i)
      }
      if (/\((mobile|tablet);[^\)]*rv:[\d\.]+\)/i.test(ua)) {
        result.firefoxos = t
      }
    }
    else if (silk) {
      result =  {
        name: 'Amazon Silk'
      , silk: t
      , version : getFirstMatch(/silk\/(\d+(\.\d+)?)/i)
      }
    }
    else if (/phantom/i.test(ua)) {
      result = {
        name: 'PhantomJS'
      , phantom: t
      , version: getFirstMatch(/phantomjs\/(\d+(\.\d+)?)/i)
      }
    }
    else if (/slimerjs/i.test(ua)) {
      result = {
        name: 'SlimerJS'
        , slimer: t
        , version: getFirstMatch(/slimerjs\/(\d+(\.\d+)?)/i)
      }
    }
    else if (/blackberry|\bbb\d+/i.test(ua) || /rim\stablet/i.test(ua)) {
      result = {
        name: 'BlackBerry'
      , blackberry: t
      , version: versionIdentifier || getFirstMatch(/blackberry[\d]+\/(\d+(\.\d+)?)/i)
      }
    }
    else if (webos) {
      result = {
        name: 'WebOS'
      , webos: t
      , version: versionIdentifier || getFirstMatch(/w(?:eb)?osbrowser\/(\d+(\.\d+)?)/i)
      };
      if( /touchpad\//i.test(ua) ){
        result.touchpad = t;
      }
    }
    else if (/bada/i.test(ua)) {
      result = {
        name: 'Bada'
      , bada: t
      , version: getFirstMatch(/dolfin\/(\d+(\.\d+)?)/i)
      };
    }
    else if (tizen) {
      result = {
        name: 'Tizen'
      , tizen: t
      , version: getFirstMatch(/(?:tizen\s?)?browser\/(\d+(\.\d+)?)/i) || versionIdentifier
      };
    }
    else if (/qupzilla/i.test(ua)) {
      result = {
        name: 'QupZilla'
        , qupzilla: t
        , version: getFirstMatch(/(?:qupzilla)[\s\/](\d+(?:\.\d+)+)/i) || versionIdentifier
      }
    }
    else if (/chromium/i.test(ua)) {
      result = {
        name: 'Chromium'
        , chromium: t
        , version: getFirstMatch(/(?:chromium)[\s\/](\d+(?:\.\d+)?)/i) || versionIdentifier
      }
    }
    else if (/chrome|crios|crmo/i.test(ua)) {
      result = {
        name: 'Chrome'
        , chrome: t
        , version: getFirstMatch(/(?:chrome|crios|crmo)\/(\d+(\.\d+)?)/i)
      }
    }
    else if (android) {
      result = {
        name: 'Android'
        , version: versionIdentifier
      }
    }
    else if (/safari|applewebkit/i.test(ua)) {
      result = {
        name: 'Safari'
      , safari: t
      }
      if (versionIdentifier) {
        result.version = versionIdentifier
      }
    }
    else if (iosdevice) {
      result = {
        name : iosdevice == 'iphone' ? 'iPhone' : iosdevice == 'ipad' ? 'iPad' : 'iPod'
      }
      // WTF: version is not part of user agent in web apps
      if (versionIdentifier) {
        result.version = versionIdentifier
      }
    }
    else if(/googlebot/i.test(ua)) {
      result = {
        name: 'Googlebot'
      , googlebot: t
      , version: getFirstMatch(/googlebot\/(\d+(\.\d+))/i) || versionIdentifier
      }
    }
    else {
      result = {
        name: getFirstMatch(/^(.*)\/(.*) /),
        version: getSecondMatch(/^(.*)\/(.*) /)
     };
   }

    // set webkit or gecko flag for browsers based on these engines
    if (!result.msedge && /(apple)?webkit/i.test(ua)) {
      if (/(apple)?webkit\/537\.36/i.test(ua)) {
        result.name = result.name || "Blink"
        result.blink = t
      } else {
        result.name = result.name || "Webkit"
        result.webkit = t
      }
      if (!result.version && versionIdentifier) {
        result.version = versionIdentifier
      }
    } else if (!result.opera && /gecko\//i.test(ua)) {
      result.name = result.name || "Gecko"
      result.gecko = t
      result.version = result.version || getFirstMatch(/gecko\/(\d+(\.\d+)?)/i)
    }

    // set OS flags for platforms that have multiple browsers
    if (!result.msedge && (android || result.silk)) {
      result.android = t
    } else if (iosdevice) {
      result[iosdevice] = t
      result.ios = t
    } else if (mac) {
      result.mac = t
    } else if (xbox) {
      result.xbox = t
    } else if (windows) {
      result.windows = t
    } else if (linux) {
      result.linux = t
    }

    // OS version extraction
    var osVersion = '';
    if (result.windowsphone) {
      osVersion = getFirstMatch(/windows phone (?:os)?\s?(\d+(\.\d+)*)/i);
    } else if (iosdevice) {
      osVersion = getFirstMatch(/os (\d+([_\s]\d+)*) like mac os x/i);
      osVersion = osVersion.replace(/[_\s]/g, '.');
    } else if (android) {
      osVersion = getFirstMatch(/android[ \/-](\d+(\.\d+)*)/i);
    } else if (result.webos) {
      osVersion = getFirstMatch(/(?:web|hpw)os\/(\d+(\.\d+)*)/i);
    } else if (result.blackberry) {
      osVersion = getFirstMatch(/rim\stablet\sos\s(\d+(\.\d+)*)/i);
    } else if (result.bada) {
      osVersion = getFirstMatch(/bada\/(\d+(\.\d+)*)/i);
    } else if (result.tizen) {
      osVersion = getFirstMatch(/tizen[\/\s](\d+(\.\d+)*)/i);
    }
    if (osVersion) {
      result.osversion = osVersion;
    }

    // device type extraction
    var osMajorVersion = osVersion.split('.')[0];
    if (
         tablet
      || nexusTablet
      || iosdevice == 'ipad'
      || (android && (osMajorVersion == 3 || (osMajorVersion >= 4 && !mobile)))
      || result.silk
    ) {
      result.tablet = t
    } else if (
         mobile
      || iosdevice == 'iphone'
      || iosdevice == 'ipod'
      || android
      || nexusMobile
      || result.blackberry
      || result.webos
      || result.bada
    ) {
      result.mobile = t
    }

    // Graded Browser Support
    // http://developer.yahoo.com/yui/articles/gbs
    if (result.msedge ||
        (result.msie && result.version >= 10) ||
        (result.yandexbrowser && result.version >= 15) ||
		    (result.vivaldi && result.version >= 1.0) ||
        (result.chrome && result.version >= 20) ||
        (result.firefox && result.version >= 20.0) ||
        (result.safari && result.version >= 6) ||
        (result.opera && result.version >= 10.0) ||
        (result.ios && result.osversion && result.osversion.split(".")[0] >= 6) ||
        (result.blackberry && result.version >= 10.1)
        || (result.chromium && result.version >= 20)
        ) {
      result.a = t;
    }
    else if ((result.msie && result.version < 10) ||
        (result.chrome && result.version < 20) ||
        (result.firefox && result.version < 20.0) ||
        (result.safari && result.version < 6) ||
        (result.opera && result.version < 10.0) ||
        (result.ios && result.osversion && result.osversion.split(".")[0] < 6)
        || (result.chromium && result.version < 20)
        ) {
      result.c = t
    } else result.x = t

    return result
  }

  var bowser = detect(typeof navigator !== 'undefined' ? navigator.userAgent : '')

  bowser.test = function (browserList) {
    for (var i = 0; i < browserList.length; ++i) {
      var browserItem = browserList[i];
      if (typeof browserItem=== 'string') {
        if (browserItem in bowser) {
          return true;
        }
      }
    }
    return false;
  }

  /**
   * Get version precisions count
   *
   * @example
   *   getVersionPrecision("1.10.3") // 3
   *
   * @param  {string} version
   * @return {number}
   */
  function getVersionPrecision(version) {
    return version.split(".").length;
  }

  /**
   * Array::map polyfill
   *
   * @param  {Array} arr
   * @param  {Function} iterator
   * @return {Array}
   */
  function map(arr, iterator) {
    var result = [], i;
    if (Array.prototype.map) {
      return Array.prototype.map.call(arr, iterator);
    }
    for (i = 0; i < arr.length; i++) {
      result.push(iterator(arr[i]));
    }
    return result;
  }

  /**
   * Calculate browser version weight
   *
   * @example
   *   compareVersions(['1.10.2.1',  '1.8.2.1.90'])    // 1
   *   compareVersions(['1.010.2.1', '1.09.2.1.90']);  // 1
   *   compareVersions(['1.10.2.1',  '1.10.2.1']);     // 0
   *   compareVersions(['1.10.2.1',  '1.0800.2']);     // -1
   *
   * @param  {Array<String>} versions versions to compare
   * @return {Number} comparison result
   */
  function compareVersions(versions) {
    // 1) get common precision for both versions, for example for "10.0" and "9" it should be 2
    var precision = Math.max(getVersionPrecision(versions[0]), getVersionPrecision(versions[1]));
    var chunks = map(versions, function (version) {
      var delta = precision - getVersionPrecision(version);

      // 2) "9" -> "9.0" (for precision = 2)
      version = version + new Array(delta + 1).join(".0");

      // 3) "9.0" -> ["000000000"", "000000009"]
      return map(version.split("."), function (chunk) {
        return new Array(20 - chunk.length).join("0") + chunk;
      }).reverse();
    });

    // iterate in reverse order by reversed chunks array
    while (--precision >= 0) {
      // 4) compare: "000000009" > "000000010" = false (but "9" > "10" = true)
      if (chunks[0][precision] > chunks[1][precision]) {
        return 1;
      }
      else if (chunks[0][precision] === chunks[1][precision]) {
        if (precision === 0) {
          // all version chunks are same
          return 0;
        }
      }
      else {
        return -1;
      }
    }
  }

  /**
   * Check if browser is unsupported
   *
   * @example
   *   bowser.isUnsupportedBrowser({
   *     msie: "10",
   *     firefox: "23",
   *     chrome: "29",
   *     safari: "5.1",
   *     opera: "16",
   *     phantom: "534"
   *   });
   *
   * @param  {Object}  minVersions map of minimal version to browser
   * @param  {Boolean} [strictMode = false] flag to return false if browser wasn't found in map
   * @param  {String}  [ua] user agent string
   * @return {Boolean}
   */
  function isUnsupportedBrowser(minVersions, strictMode, ua) {
    var _bowser = bowser;

    // make strictMode param optional with ua param usage
    if (typeof strictMode === 'string') {
      ua = strictMode;
      strictMode = void(0);
    }

    if (strictMode === void(0)) {
      strictMode = false;
    }
    if (ua) {
      _bowser = detect(ua);
    }

    var version = "" + _bowser.version;
    for (var browser in minVersions) {
      if (minVersions.hasOwnProperty(browser)) {
        if (_bowser[browser]) {
          // browser version and min supported version.
          return compareVersions([version, minVersions[browser]]) < 0;
        }
      }
    }

    return strictMode; // not found
  }

  /**
   * Check if browser is supported
   *
   * @param  {Object} minVersions map of minimal version to browser
   * @param  {Boolean} [strictMode = false] flag to return false if browser wasn't found in map
   * @param  {String}  [ua] user agent string
   * @return {Boolean}
   */
  function check(minVersions, strictMode, ua) {
    return !isUnsupportedBrowser(minVersions, strictMode, ua);
  }

  bowser.isUnsupportedBrowser = isUnsupportedBrowser;
  bowser.compareVersions = compareVersions;
  bowser.check = check;

  /*
   * Set our detect method to the main bowser object so we can
   * reuse it to test other user agents.
   * This is needed to implement future tests.
   */
  bowser._detect = detect;

  return bowser
});

(function($){
  UABBTrigger = {

      /**
       * Trigger a hook.
       *
       * @since 1.1.0.3
       * @method triggerHook
       * @param {String} hook The hook to trigger.
       * @param {Array} args An array of args to pass to the hook.
       */
      triggerHook: function( hook, args )
      {
        $( 'body' ).trigger( 'uabb-trigger.' + hook, args );
      },
    
      /**
       * Add a hook.
       *
       * @since 1.1.0.3
       * @method addHook
       * @param {String} hook The hook to add.
       * @param {Function} callback A function to call when the hook is triggered.
       */
      addHook: function( hook, callback )
      {
        $( 'body' ).on( 'uabb-trigger.' + hook, callback );
      },
    
      /**
       * Remove a hook.
       *
       * @since 1.1.0.3
       * @method removeHook
       * @param {String} hook The hook to remove.
       * @param {Function} callback The callback function to remove.
       */
      removeHook: function( hook, callback )
      {
        $( 'body' ).off( 'uabb-trigger.' + hook, callback );
      },
  };
})(jQuery);

jQuery(document).ready(function( $ ) {

    if( typeof bowser !== 'undefined' && bowser !== null ) {

      var uabb_browser   = bowser.name,
          uabb_browser_v = bowser.version,
          uabb_browser_class = uabb_browser.replace(/\s+/g, '-').toLowerCase(),
          uabb_browser_v_class = uabb_browser_class + parseInt( uabb_browser_v );
      
      $('html').addClass(uabb_browser_class).addClass(uabb_browser_v_class);
      
    }

    $('.uabb-row-separator').parents('html').css('overflow-x', 'hidden');
});

;/**
 * This file should contain frontend logic for 
 * all module instances.
 */(function($) {

    if($(window).width() <= 768 && $(window).width() >= 481 ) {
        $('.fl-node-imew2zsl7qh9 .pp-heading-separator, .fl-node-imew2zsl7qh9 .pp-heading').removeClass('pp-center');
        $('.fl-node-imew2zsl7qh9 .pp-heading-separator, .fl-node-imew2zsl7qh9 .pp-heading').addClass('pp-tablet-');
    }

    if( $(window).width() <= 480 ) {
        $('.fl-node-imew2zsl7qh9 .pp-heading-separator, .fl-node-imew2zsl7qh9 .pp-heading').removeClass('pp-center');
        $('.fl-node-imew2zsl7qh9 .pp-heading-separator, .fl-node-imew2zsl7qh9 .pp-heading').addClass('pp-mobile-');
    }

})(jQuery);
jQuery(function($) {
	
		$(function() {
		$( '.fl-node-rxj5n430apg8 .fl-photo-img' )
			.on( 'mouseenter', function( e ) {
				$( this ).data( 'title', $( this ).attr( 'title' ) ).removeAttr( 'title' );
			} )
			.on( 'mouseleave', function( e ){
				$( this ).attr( 'title', $( this ).data( 'title' ) ).data( 'title', null );
			} );
	});
		window._fl_string_to_slug_regex = 'a-zA-Z0-9';
});

;/**
 * This file should contain frontend logic for 
 * all module instances.
 */
;(function($) {

	PPContentGrid = function(settings) {
		this.settings       = settings;
		this.nodeClass      = '.fl-node-' + settings.id;
		this.wrapperClass   = this.nodeClass + ' .pp-content-post-' + this.settings.layout;
		this.postClass      = this.wrapperClass + ' .pp-content-' + this.settings.layout + '-post';
		this.matchHeight	= settings.matchHeight == 'yes' ? true : false;
		this.style			= settings.style;
		this.masonry		= settings.masonry == 'yes' ? true : false;
		this.filters 		= settings.filters;
		this.filterTax 		= settings.filterTax;
		this.filterType 	= settings.filterType;
		this.page           = $( this.wrapperClass ).data('paged');
		this.isFiltering	= false;
		this.isDefaultFilter = false;
		this.activeFilter 	= '';
		this.totalPages 	= 1;
		this.currentPage 	= 1;
		this.cacheData		= {};
		this.ajaxData 		= {};
		this.includeSettings = true;

		if ( this._hasPosts() ) {
			this._initLayout();
			this._initSearch();
		}
	};

	PPContentGrid.prototype = {
		settings        : {},
		nodeClass       : '',
		wrapperClass    : '',
		postClass       : '',
		filters			: false,
		filterTax		: '',
		filterType		: '',
		filterData		: {},
		isFiltering		: false,
		isDefaultFilter	: false,
		activeFilter	: '',
		totalPages		: 1,
		currentPage		: 1,
		cacheData		: {},
		ajaxData 		: {},
		includeSettings	: true,
		matchHeight		: false,
		masonry			: false,
		style			: '',

		_hasPosts: function() {
			return $(this.postClass).length > 0;
		},

		_useCssGrid: function() {
			return this.matchHeight && 'dynamic' === this.filterType;
		},

		_initIsotopeData: function() {
			var postFilterData = {
				itemSelector: '.pp-content-post',
				percentPosition: true,
				transitionDuration: '0.3s',
				hiddenStyle: {
					opacity: 0
				},
				visibleStyle: {
					opacity: 1
				},
				masonry: {
					columnWidth: '.pp-content-post',
					gutter: '.pp-grid-space'
				},
				isOriginLeft: ! $('body').hasClass( 'rtl' ),
			};

			if ( ! this.masonry ) {
				postFilterData = $.extend( {}, postFilterData, {
					layoutMode: 'fitRows',
					fitRows: {
						gutter: '.pp-grid-space'
				  	},
				} );
			}

			// set filter data globally to use later for ajax scroll pagination.
			this.filterData = postFilterData;

			// set default filter as active filter.
			if ( this.settings.defaultFilter !== '' ) {
				this.activeFilter = this.settings.defaultFilter;
			}
		},

		_initSearch: function() {
			var searchFormClass = 'cg-search-form';
			
			$(this.nodeClass).trigger( 'grid.searchInit', [ searchFormClass, this ] );

			if ( ! $( this.nodeClass ).hasClass( 'cg-search-result' ) && $( '.' + searchFormClass ).length > 0 ) {
				return;
			}

			$( '.' + searchFormClass ).find( 'form' ).on( 'submit', function(e) {
				e.preventDefault();
				
				var input = $(e.target).find('input[type="search"]');
		
				bb_powerpack.search_term = input.val();
				if ( input.val() !== '' ) {
					this.settings.is_search = true;
					this.isFiltering = true;
				} else {
					this.settings.is_search = false;
					this.isFiltering = false;
					this.cacheData = {};
					this.ajaxData = {};
				}
				this._getAjaxPosts( this.activeFilter, this.filterData, this.currentPage );
			}.bind( this ) );
		
			$('.' + searchFormClass).find( 'form input' ).on( 'input clear', function(e) {
				if ( $(e.target).val() === '' ) {
					bb_powerpack.search_term = '';
					this.settings.is_search = false;
					this.isFiltering = false;
					this.cacheData = {};
					this.ajaxData = {};
					this._getAjaxPosts( this.activeFilter, this.filterData, this.currentPage );
				}
			}.bind( this ) );
		},

		_initLayout: function() {
			if ( $(this.nodeClass).find('.pp-posts-wrapper').hasClass('pp-posts-initiated') ) {
				return;
			}

			switch(this.settings.layout) {

				case 'grid':
					this._initIsotopeData();
					this._gridLayout();
					this._initPagination();
					this._reLayout();
					break;

				case 'carousel':
					this._carouselLayout();
					break;
			}

			$(this.postClass).css('visibility', 'visible');

			var self = this;

			$(window).on('load', function() {
				if ( ! $( self.nodeClass ).hasClass( 'pp-cg-no-page-scroll' ) ) {
					FLBuilderLayout._scrollToElement( $( self.nodeClass + ' .pp-paged-scroll-to' ) );
				}
			});

			$(this.nodeClass).on( 'grid.afterInit carousel.afterInit', function() {
				$(self.nodeClass).find('.pp-posts-wrapper').addClass('pp-posts-initiated');

				if ( 'carousel' === self.settings.layout ) {
					setTimeout(function() {
						self._gridLayoutMatchHeightSimple();
					}, 200);
				}
				if ( $(self.nodeClass).find( '.owl-nav' ).length > 0 && self.settings.carousel.autoplay ) {
					var carousel = $(self.nodeClass).find( '.owl-carousel' );
					$(self.nodeClass).find( '.owl-nav button' ).on( 'mouseover', function() {
						carousel.trigger( 'stop.owl.autoplay' );
					} ).on( 'mouseleave', function() {
						carousel.trigger( 'play.owl.autoplay' );
					} );
				}
			} );

			// Fix native lazy load issue.
			$(this.nodeClass).find('.pp-posts-wrapper img').on('load', function() {
				if ( 'lazy' !== $(this).attr('loading') ) {
					return;
				}
				var postsWrapper = $(self.nodeClass).find('.pp-content-post-grid');
				if ( 'undefined' !== typeof $.fn.isotope ) {
					setTimeout(function() {
						if ( postsWrapper.data( 'isotope' ) ) {
							postsWrapper.isotope('layout');
						}
					}, 500);
				}
			});
		},

		_gridLayout: function()
		{
			var wrap = $(this.wrapperClass);
			var postFilterData = this.filterData;

			this._loadItemsFromCache();

			this._setLastPostClass();
			this._setCenterAlign();

			$( window ).on( 'resize', this._setCenterAlign.bind( this ) );

			wrap.imagesLoaded( function() {

				var node = $(this.nodeClass);
				var base = this;
				var postFilters = ! this._useCssGrid() && ! wrap.data( 'isotope' ) ? wrap.isotope( postFilterData ) : wrap;

                if ( this.settings.filters || this.masonry ) {

					var filterWrap = $(this.nodeClass).find('.pp-post-filters');
					var filterToggle = $(this.nodeClass).find('.pp-post-filters-toggle');

					filterToggle.on('click', function () {
						filterWrap.slideToggle(function () {
							if ($(this).is(':visible')) {
								filterToggle.addClass('pp-post-filters-open');
							}
							if (!$(this).is(':visible')) {
								filterToggle.removeClass('pp-post-filters-open');
							}
						});
					});

					filterWrap.on('click keyup', '.pp-post-filter', function(e) {
						var shouldFilter = 'click' === e.type || ( 'keyup' === e.type && ( 13 === e.keyCode || 13 === e.which ) )
						if ( ! shouldFilter ) {
							return;
						}
						// set active filter globally to use later for ajax scroll pagination.
						base.activeFilter = $(this).data('term');
						base.isFiltering = true;

						if ('static' === base.filterType) {
							var filterVal = $(this).attr('data-filter');
							postFilters.isotope({ filter: filterVal });
						} else {
							var term = $(this).data('term');
							$(base.wrapperClass).addClass('pp-is-filtering');
							base._getPosts(term, postFilterData);
						}

						filterWrap.find('.pp-post-filter').removeClass('pp-filter-active');
						$(this).addClass('pp-filter-active');

						filterToggle.find('span.toggle-text').html($(this).text());
						if (filterToggle.hasClass('pp-post-filters-open')) {
							filterWrap.slideUp();
							filterToggle.removeClass('pp-post-filters-open');
						}

						$(base.nodeClass).trigger('grid.filter.change');
					});

					$( window ).on( 'resize', function() {
						if ( ! filterToggle.is( ':visible' ) ) {
							filterWrap.removeAttr( 'style' );
						}
					} );

					// Default filter.
					if ( base.settings.defaultFilter !== '' ) {
						var defaultFilter = base.settings.defaultFilter;
						if ( filterWrap.find('li[data-term="' + defaultFilter + '"]').length > 0 ) {
							self.isDefaultFilter = true;
							filterWrap.find('li[data-term="' + defaultFilter + '"]').trigger('click');
						}
					}

					// Trigger filter by hash parameter in URL.
					if ( '' !== location.hash ) {
						var filterHash = location.hash.split('#')[1];

						filterWrap.find('li[data-term="' + filterHash + '"]').trigger('click');
					}

					// Trigger filter on hash change in URL.
					$(window).on('hashchange', function() {
						if ( '' !== location.hash ) {
							var filterHash = location.hash.split('#')[1];
	
							filterWrap.find('li[data-term="' + filterHash + '"]').trigger('click');
						}
					});
                }

				if ('dynamic' === base.filterType) {
					$(base.nodeClass).find('.fl-builder-pagination a').off('click').on('click', function (e) {
						e.preventDefault();
						var pageNumber = base._getPageNumber( this );
						base.currentPage = pageNumber;
						base._getPosts('', postFilterData, pageNumber);
					});
				}

                if( ! this.masonry ) {
                    setTimeout( function() {
						if ( base.settings.filters && 'static' === base.filterType ) {
							node.find('.pp-filter-active').trigger('click');
						}
						base._gridLayoutMatchHeight();
						if ( wrap.data( 'isotope' ) ) {
							wrap.isotope('layout');
						}
                    }, 1000 );
                }

				$(this.nodeClass).trigger('grid.afterInit');

			}.bind( this ) );
		},

		_loadItemsFromCache: function() {
			if ( window.localStorage && $(this.nodeClass).hasClass( 'cg-cache-active' ) ) {
				var cachedResponse = window.localStorage.getItem( 'ppcg_' + this.settings.id );
				if ( cachedResponse ) {
					cachedResponse = JSON.parse( cachedResponse );
					cachedTime = cachedResponse.timestamp;
					timeDiff = ( new Date().getTime() - cachedTime ) / 1000;
					timeDiff /= (60*60);
					if ( timeDiff >= 12 || location.href !== cachedResponse.currentUrl ) { // TODO
						window.localStorage.removeItem( 'ppcg_' + this.settings.id );
					} else {
						var pages = cachedResponse.cacheData.ajaxCache[cachedResponse.filter].page;
						Object.keys( pages ).forEach(function(page) {
							this._renderPosts(pages[page], {
								term: cachedResponse.filter,
								isotopeData: this.filterData,
								page: cachedResponse.paged
							});
						}.bind(this));

						this.currentPage  = cachedResponse.currentPage;
						this.activeFilter = cachedResponse.activeFilter;
						this.cacheData    = cachedResponse.cacheData;

						$(this.nodeClass).find('.pp-post-filter').removeClass('pp-filter-active');

						if ( '' === this.activeFilter || 'all' === cachedResponse.filter ) {
							$(this.nodeClass).find('.pp-post-filter[data-filter="*"]').addClass('pp-filter-active');
						} else {
							$(this.nodeClass).find('.pp-post-filter[data-term="' + this.activeFilter + '"]').addClass('pp-filter-active');
						}
					}
				}
			}
		},
		
		_removeItemsFromCache: function() {
			if ( window.localStorage ) {
				window.localStorage.removeItem( 'ppcg_' + this.settings.id );
			}
		},

		_setLastPostClass: function() {
			var posts = $( this.postClass );

			if ( posts.length > 0 ) {
				var last = posts[ posts.length - 1 ];

				$( last ).addClass( 'pp-post-last' );
			}
		},

		_setCenterAlign: function() {
			if ( $( this.nodeClass ).hasClass( 'cg-center-align' ) ) {
				var columns = this._getColumns();
				var gridSpace = $( this.wrapperClass ).find( '.pp-grid-space' ).outerWidth();
				if ( $( this.postClass ).length >= columns ) {
					gridSpace = gridSpace * ( columns - 1 );
				}
				if ( 1 === columns ) {
					gridSpace = 0;
				}
				var colWidth = ( $( this.nodeClass + ' .fl-module-content' ).outerWidth() / columns ) - gridSpace;

				if ( colWidth > 0 ) {
					this.filterData.masonry.columnWidth = colWidth;
					this.filterData.masonry.isFitRows = true;
					this.filterData.percentPosition = false;

					gridSpace += 0.18;

					if ( $( this.postClass ).length < columns ) {
						var wrapperWidth = ( colWidth * $( this.postClass ).length ) + ( gridSpace * ( $( this.postClass ).length - 1 ) );
					} else {
						var wrapperWidth = ( colWidth * columns ) + gridSpace;
					}

					$( this.postClass ).css( 'width', colWidth + 'px' );
					$( this.wrapperClass ).css( 'width', wrapperWidth + 'px' );
				}
			}
		},

		_carouselLayout: function()
		{
			var self = this;
			var wrap = $(this.nodeClass + ' .pp-content-post-carousel .pp-content-posts-inner');

			var setAriaAttrs = function(e) {
				var visibleSlides = 0; // Counter for real visible slides
				$(e.target).find('.owl-item').each(function(index) {
					// Check if the item is not a clone
					if ( ! $(this).hasClass('cloned') ) {
						visibleSlides++;
						$(this).attr('aria-label', 'Item ' + visibleSlides + ' of ' + e.item.count);
					}
				});

				var currentIndex = e.item.index; // Get the index of the first visible slide
				var visibleItems = e.page.size;  // Get the number of visible slides

				// Loop through all slides
				$(e.target).find('.owl-item').each(function(index) {
					// Set aria-hidden="false" for visible slides, aria-hidden="true" for non-visible slides
					if (index >= currentIndex && index < currentIndex + visibleItems) {
						$(this).attr('aria-hidden', 'false');
						$(this).attr('tabindex', '0');
						$(this).find('a, button').attr('tabindex', '0');
					} else {
						$(this).attr('aria-hidden', 'true');
						$(this).attr('tabindex', '-1');
						$(this).find('a, button').attr('tabindex', '-1');
					}
				});
			};

			var owlOptions = {
				onInitialized: function(e) {
					var count = 1;
					$(e.target).find('.owl-dot').each(function() {
						$(this).append( '<span class="sr-only">Slide group ' + count + '</span>' );
						count++;
						if ( $(this).hasClass( 'active' ) ) {
							$(this).attr('aria-current', 'true');
						} else {
							$(this).attr('aria-current', 'false');
						}
					});
					
					$(e.target).find( '.owl-item' ).attr('role','group');

					setAriaAttrs(e);

					self._gridLayoutMatchHeightSimple();
				},
				onChanged: function(e) {
					$(e.target).find('.owl-dot').each(function() {
						if ( $(this).hasClass( 'active' ) ) {
							$(this).attr('aria-current', 'true');
						} else {
							$(this).attr('aria-current', 'false');
						}
					});
			
					//var currentItem = e.item.index + 1;
					//$(e.target).find('.owl-item').eq(currentItem).focus();

					setAriaAttrs(e);
				},
				onResized: this._gridLayoutMatchHeightSimple.bind( this ),
				onRefreshed: this._gridLayoutMatchHeightSimple.bind( this ),
				onLoadedLazy: this._gridLayoutMatchHeightSimple.bind( this ),
				rtl: $('body').hasClass( 'rtl' )
			};
			if ( $(this.postClass).length < this.settings.carousel.items ) {
				this.settings.carousel.slideBy = 'page';
				this.settings.carousel.loop = false;
			}
			if ( this.settings.carousel.lazyLoad ) {
				$( this.postClass ).each(function() {
					if ( $( this ).find( '.pp-content-grid-post-image img' ).length > 0 ) {
						var src = $( this ).find( '.pp-content-grid-post-image img' ).addClass( 'owl-lazy' ).attr( 'src' );
						$( this ).find( '.pp-content-grid-post-image img' ).removeAttr( 'src' ).attr( 'data-src', src );
					} else {
						var src = $( this ).find( '.pp-post-image img' ).addClass( 'owl-lazy' ).attr( 'src' );
						$( this ).find( '.pp-post-image img' ).removeAttr( 'src' ).attr( 'data-src', src );
					}
				});
			}

			owlOptions = $.extend({}, this.settings.carousel, owlOptions);
			$(this.nodeClass).trigger( 'carousel.beforeInit', [ owlOptions ] );

			wrap.imagesLoaded( function() {
				wrap.owlCarousel( owlOptions );

				var owl = wrap.data('owl.carousel');

				wrap.on( 'keyup', function(e) {
					if ( 37 === e.keyCode || 37 === e.which ) {
						owl.prev();
					}
					if ( 39 === e.keyCode || 39 === e.which ) {
						owl.next();
					}
				} );

				if ( $(this.nodeClass).hasClass( 'overflow-visible' ) ) {
					$( this.nodeClass ).closest('.fl-row').css( 'overflow-x', 'hidden' );
					$( this.nodeClass ).find('.owl-carousel').css( 'overflow', 'visible' );
				}

				$(this.nodeClass).trigger('carousel.afterInit');
			}.bind( this ) );
		},

		_getPosts: function (term, isotopeData, paged) {
			var processAjax = false,
				filter 		= term,
				paged 		= (!paged || 'undefined' === typeof paged) ? 1 : paged;

			if ('undefined' === typeof term || '' === term) {
				filter = 'all';
			}

			this._getTotalPages();

			var cacheData = this._getCacheData(filter);

			if ('undefined' === typeof cacheData) {
				processAjax = true;
			} else {
				var cachedResponse = cacheData.page[paged];
				if ('undefined' === typeof cachedResponse) {
					processAjax = true;
				} else {
					$(this.nodeClass).trigger('grid.beforeRender');
					this._renderPosts(cachedResponse, {
						term: term,
						isotopeData: isotopeData,
						page: paged
					});
				}
			}

			if (processAjax) {
				this._getAjaxPosts(term, isotopeData, paged);
			}
		},

		_getAjaxPosts: function (term, isotopeData, paged) {
			var taxonomy = this.filterTax,
				paged = 'undefined' === typeof paged ? false : paged,
				self = this;

			var currentPage = bb_powerpack.current_page.split('?')[0];

			var data = {
				pp_action: 'get_ajax_posts',
				node_id: this.settings.id,
				paged: ! paged ? this.page : paged,
				current_page: currentPage,
			};

			if ( 'undefined' !== typeof this.includeSettings && this.includeSettings ) {
				data.settings = this.settings.fields;
			}

			if ( $(this.wrapperClass).data('template-id') ) {
				data.template_id = $(this.wrapperClass).data('template-id');
				data.template_node_id = $(this.wrapperClass).data('template-node-id');
			}

			// Archive.
			if ( this.settings.is_archive ) {
				data['is_archive'] = true;
			}

			// Term.
			if ('undefined' !== typeof term && '' !== term) {
				data['term'] = term;
			} else if ( this.settings.is_tax && this.settings.current_term ) {
				data['is_tax'] = true;
				data['taxonomy'] = this.settings.current_tax;
				data['term'] = this.settings.current_term;
			}
			// Term - provide current term data additionally.
			if ( this.settings.is_tax && this.settings.current_term ) {
				data['current_tax'] = this.settings.current_tax;
				data['current_term'] = this.settings.current_term;
			}

			// Author.
			if ( this.settings.is_author && this.settings.current_author ) {
				data['is_author'] = true;
				data['author_id'] = this.settings.current_author;
			}

			if ( this.settings.is_search ) {
				data['is_search'] = true;
				data['search_term'] = bb_powerpack.search_term;
			}

			if ( $(this.wrapperClass).data('orderby') ) {
				data['orderby'] = $(this.wrapperClass).data('orderby');
			}

			this.ajaxData = data;

			if ( ! this.paginating ) {
				$(this.wrapperClass).addClass('pp-is-filtering');
			}

			$.post(
				location.href.split('#').shift(),//.replace( location.search, '' ),
				data,
				function (response) {
					// if ( 'undefined' !== typeof response.success && ! response.success ) {
					// 	self.includeSettings = true;
					// 	self._getAjaxPosts( term, isotopeData, paged );
					// 	return;
					// }
					self.includeSettings = true;
					self._setCacheData(term, response, paged);
					$(self.nodeClass).trigger('grid.beforeRender');
					self._renderPosts(response, {
						term: term,
						isotopeData: isotopeData,
						page: paged
					});
					$(self.wrapperClass).removeClass('pp-is-filtering');
					if ( ! self.paginating ) {
						self.paginating = false;
					}
				}
			).fail( function( xhr ) {
				if ( 404 == xhr.status && 'undefined' !== xhr.responseJSON.data ) {
					var response = xhr.responseJSON;
					self.includeSettings = true;
					self._setCacheData(term, response, paged);
					$(self.nodeClass).trigger('grid.beforeRender');
					self._renderPosts(response, {
						term: term,
						isotopeData: isotopeData,
						page: paged
					});
					$(self.wrapperClass).removeClass('pp-is-filtering');
					if ( ! self.paginating ) {
						self.paginating = false;
					}
				}
			} );
		},

		_renderPosts: function (response, args) {
			var self = this,
				wrap = $(this.wrapperClass),
				posts = $(response.data),
				count = 0;

			posts.each(function() {
				if ( $(this).hasClass('pp-content-post') ) {
					count++;
				}
			});

			wrap.parent().css( '--items-count', count );

			if ( ( 'load_more' !== self.settings.pagination && 'scroll' !== self.settings.pagination ) || self.isFiltering ) {
				if ( wrap.data( 'isotope' ) ) {
					wrap.isotope('remove', $(this.postClass));
				} else {
					wrap.find( this.postClass ).remove();
				}
			}

			if ( wrap.data( 'isotope' ) ) {
				wrap.isotope( 'remove', $( '.pp-posts-not-found-text' ) );
			} else {
				wrap.find( '.pp-posts-not-found-text' ).remove();
			}

			if ( wrap.data( 'isotope' ) ) {
				if (!this.masonry) {
					wrap.isotope('insert', posts, this._gridLayoutMatchHeight.bind( this ) );
					wrap.imagesLoaded( function () {
						setTimeout(function () {
							self._gridLayoutMatchHeight();
						}, 150);
					}.bind( this ) );
				} else {
					wrap.isotope('insert', posts);
				}

				wrap.find('.pp-grid-space').remove();
				wrap.append('<div class="pp-grid-space"></div>');
			} else {
				posts.hide();
				if ( ( 'load_more' == self.settings.pagination || 'scroll' == self.settings.pagination ) && ! self.isFiltering ) {
					wrap.append( posts.fadeIn(400) );
				} else {
					wrap.html( posts.fadeIn(400) );
				}
			}

			wrap.imagesLoaded( function () {
				setTimeout(function () {
					if (!this.masonry) {
						self._gridLayoutMatchHeight();
					}
					if ( wrap.data( 'isotope' ) ) {
						wrap.isotope('layout');
					}
				}, 500);
			}.bind( this ) );

			if ( 'load_more' === self.settings.pagination ) {
				$(self.nodeClass).find('.pp-content-grid-load-more').remove();
			}
			if ( 'scroll' === self.settings.pagination ) {
				$(self.nodeClass).find('.pp-content-grid-loader').remove();
			}

			if (response.pagination) {
				var $pagination = $(response.pagination);

				if ( 'numbers' === self.settings.pagination ) {
					$pagination.removeAttr( 'style' );
				}

				$(self.nodeClass).find('.fl-builder-pagination').remove();
				$(self.nodeClass).find( '> .fl-module-content' ).append($pagination);
				if ( 'load_more' === self.settings.pagination ) {
					setTimeout(function() {
						self._getTotalPages();
						if ( self.totalPages !== self.currentPage ) {
							$(self.nodeClass).find('.fl-module-content .pp-content-grid-load-more').fadeIn();
							self._initPagination();
						} else {
							$(self.nodeClass).find('.fl-module-content .pp-content-grid-load-more').slideUp();
						}
					}, 250);
				}
				$(self.nodeClass).off('click', '.pp-ajax-pagination a').on('click', '.pp-ajax-pagination a', function (e) {
					e.preventDefault();
					var pageNumber = self._getPageNumber( this );
					self.currentPage = pageNumber;
					self._getPosts(args.term, args.isotopeData, pageNumber);
				});
			} else {
				$(self.nodeClass).find('.fl-builder-pagination').remove();
			}

			if ( ('load_more' !== self.settings.pagination && 'scroll' !== self.settings.pagination) || self.isFiltering ) {
				if ( self.settings.scrollTo && ! self.isDefaultFilter ) {
					var scrollToClass = self.settings.scrollToClass || self.nodeClass;
					var offsetTop = $(scrollToClass).offset().top - 200;
					$('html, body').stop().animate({
						scrollTop: offsetTop
					}, 300);
					self.isDefaultFilter = false;
				}
			}

			// Append any scripts and styles found in response.
			if ( response.styles_scripts && '' !== response.styles_scripts ) {
				$('body').append( $( response.styles_scripts ) );
			}

			if ( self.isFiltering ) {
				self.isFiltering = false;
				$(self.nodeClass).trigger( 'grid.filter.complete' );
			}
			wrap.removeClass('pp-is-filtering');

			$(self.nodeClass).trigger('grid.rendered');
		},

		_getPageNumber: function( pageElement )
		{
			var pageNumber = parseInt( $(pageElement).text() ); //$(pageElement).attr('href').split('#page-')[1];

			if ( $(pageElement).hasClass('next') ) {
				pageNumber = parseInt( $(pageElement).parents('.pp-content-grid-pagination').find('.current').text() ) + 1;
			}
			if ( $(pageElement).hasClass('prev') ) {
				pageNumber = parseInt( $(pageElement).parents('.pp-content-grid-pagination').find('.current').text() ) - 1;
			}

			return pageNumber;
		},

		_setCacheData: function (filter, response, paged) {
			if ('undefined' === typeof filter || '' === filter) {
				filter = 'all';
			}
			if ('undefined' === typeof paged || !paged) {
				paged = 1;
			}

			if ('undefined' === typeof this.cacheData.ajaxCache) {
				this.cacheData.ajaxCache = {};
			}
			if ('undefined' === typeof this.cacheData.ajaxCache[filter]) {
				this.cacheData.ajaxCache[filter] = {};
			}
			if ('undefined' === typeof this.cacheData.ajaxCache[filter].page) {
				this.cacheData.ajaxCache[filter].page = {};
			}

			this.cacheData.ajaxCache[filter].page[paged] = response;

			if ( window.localStorage ) {
				var localData = {
					paged: paged,
					currentPage: this.currentPage,
					currentUrl: location.href,
					cacheData: this.cacheData,
					filter: filter,
					activeFilter: this.activeFilter,
					timestamp: new Date().getTime()
				};
				window.localStorage.setItem( 'ppcg_' + this.settings.id, JSON.stringify(localData) );
			}
		},

		_getCacheData: function (filter) {
			var cacheData = this.cacheData;

			if ('undefined' === typeof cacheData.ajaxCache) {
				cacheData.ajaxCache = {};
			}

			return cacheData.ajaxCache[filter];
		},

		_getColumns: function() {
			var columns = this.settings.postColumns.xl;

			if (window.innerWidth <= this.settings.breakpoints.large) {
				columns = this.settings.postColumns.lg;
			}
			if (window.innerWidth <= this.settings.breakpoints.medium) {
				columns = this.settings.postColumns.md;
			}
			if (window.innerWidth <= this.settings.breakpoints.small) {
				columns = this.settings.postColumns.sm;
			}

			return columns;
		},

		_gridLayoutMatchHeight: function()
		{
			var highestBox = 0;
			var contentHeight = 0;
			var postElements = $(this.postClass + ':visible');
			var columns = this._getColumns();

			if (! this.matchHeight || 1 === columns) {
				return;
			}

			if ( 'style-9' === this.style ) {
				return;
			}

			if ( this._useCssGrid() ) {
				return;
			}

			if ( this.settings.layout === 'grid' ) {
				if ( this.masonry ) {
					return;
				}

				postElements.css('height', 'auto');

				var rows = Math.round(postElements.length / columns);

				if ( postElements.length % columns > 0 ) {
					rows = rows + 1;
				}

				// range.
				var j = 1,
					k = columns;

				for( var i = 0; i < rows; i++ ) {
					// select number of posts in the current row.
					var postsInRow = $(this.postClass + ':visible:nth-child(n+' + j + '):nth-child(-n+' + k + ')');

					// get height of the larger post element within the current row.
					postsInRow.css('height', '').each(function () {
						if ($(this).height() > highestBox) {
							highestBox = $(this).height();
							contentHeight = $(this).find('.pp-content-post-data').outerHeight();
						}
					});
					// apply the height to all posts in the current row.
					postsInRow.height(highestBox);

					// increment range.
					j = k + 1;
					k = k + columns;
					if ( k > postElements.length ) {
						k = postElements.length;
					}
					highestBox = 0;
				}
			} else {
				// carousel layout.
				postElements.css('height', '').each(function(){

					if($(this).height() > highestBox) {
						highestBox = $(this).height();
						contentHeight = $(this).find('.pp-content-post-data').outerHeight();
					}
				});

				postElements.height(highestBox);
			}
            //$(this.postClass).find('.pp-content-post-data').css('min-height', contentHeight + 'px').addClass('pp-content-relative');
		},

		_gridLayoutMatchHeightSimple: function () {
			if ( ! this.matchHeight ) {
				return;
			}

			if ( $(this.wrapperClass).hasClass( 'pp-css-grid' ) ) {
				return;
			}

			if ( 'style-9' === this.style ) {
				return;
			}

			var highestBox = 0;
			var contentHeight = 0;
			var postElements = $(this.postClass);

			var columns = this._getColumns();

			if ( 1 === columns && this.settings.layout === 'grid' ) {
				return;
			}

			postElements.css('height', '').each(function () {

				if ($(this).height() > highestBox) {
					highestBox = $(this).height();
					contentHeight = $(this).find('.pp-content-post-data').outerHeight();
				}
			});

			postElements.height(highestBox);
		},

		_initPagination: function()
		{
			var self = this;

			setTimeout(function() {
				self._getTotalPages();

				if ( self.settings.pagination === 'load_more' ) {
					self._initLoadMore();
				}
				if ( self.settings.pagination === 'scroll' && typeof FLBuilder === 'undefined' ) {
					self._initScroll();
				}
			}, 500);
		},

		_getTotalPages: function()
		{
			var pages = $( this.nodeClass + ' .pp-content-grid-pagination' ).find( 'li .page-numbers:not(.next)' );

			if ( pages.length > 1) {
				var total = pages.last().text().replace( /\D/g, '' )
				this.totalPages = parseInt( total );
			} else {
				this.totalPages = 1;
			}

			return this.totalPages;
		},

		_initLoadMore: function()
		{
			var self 		= this,
				$button 	= $(this.nodeClass).find('.pp-grid-load-more-button'),
				currentPage = self.currentPage,
				activeFilter = self.activeFilter,
				isAjaxPagination = 'dynamic' === self.filterType;

			if ( ! self.filters || 'dynamic' !== self.filterType ) {
				activeFilter = '';
			}

			$(self.nodeClass).on('grid.filter.change', function() {
				// re-assign active filter.
				if ( self.filters && 'dynamic' === self.filterType ) {
					activeFilter = self.activeFilter
				}
			});

			$button.off('click').on('click', function(e) {
				e.preventDefault();

				$(this).addClass('disabled loading');
				self.isFiltering = false;
				self.paginating = true;

				currentPage = parseInt( currentPage ) + 1;

				self._getPosts(activeFilter, self.filterData, currentPage);
				self.currentPage = currentPage;
			});

			$(self.nodeClass).on('grid.rendered', function() {
				$button.removeClass( 'disabled loading' );

				if ( currentPage >= self.totalPages ) {
					$button.parent().hide();
				}
			});

			// Reset pagination index on filter.
			$(self.nodeClass).on('grid.filter.complete', function() {
				if ( $(self.nodeClass).find( '.pp-content-grid-pagination' ).length > 0 ) {
					self._getTotalPages();
					self.currentPage = currentPage = 1;
					$button.parent().show();
				} else {
					$button.parent().hide();
				}
			});
		},

		_initScroll: function()
		{
			var	self			= this,
				gridOffset 		= $(this.wrapperClass).offset(),
				gridHeight		= $(this.wrapperClass).height(),
				winHeight		= $(window).height(),
				currentPage 	= this.currentPage,
				activeFilter	= self.activeFilter,
				rendered		= false,
				loaded			= false;
			
			if ( self.scrollPaginationIninitiated ) {
				return;
			}

			if ( ! self.filters || 'dynamic' !== self.filterType ) {
				activeFilter = '';
			}

			$(window).on('scroll', function() {
				if ( loaded ) {
					return;
				}
				var scrollPos = $(window).scrollTop();

				if ( scrollPos >= gridOffset.top - ( winHeight - gridHeight ) ) {
					self.isFiltering = false;
					currentPage = parseInt( currentPage ) + 1;

					$(self.nodeClass).find('.pp-content-grid-loader').show();

					if ( currentPage <= self.totalPages ) {
						loaded = true;
						self.paginating = true;
						self._getPosts(activeFilter, self.filterData, currentPage);
					} else {
						loaded = true;
						$(self.nodeClass).find('.pp-content-grid-loader').hide();
					}

					self.currentPage = currentPage;
				}
			}.bind( this ) );

			$(self.nodeClass).on('grid.filter.change', function() {
				// re-assign active filter.
				if ( self.filters && 'dynamic' === self.filterType ) {
					activeFilter = self.activeFilter
				}
				
				// get container height.
				gridHeight = $(self.wrapperClass).height();
				self._gridLayoutMatchHeightSimple();

				if ( $(self.wrapperClass).data( 'isotope' ) ) {
					$(self.wrapperClass).isotope('layout');
				}

				if ( 'dynamic' === self.filterType ) {
					self._getTotalPages();
					self.currentPage = currentPage = 1;
					loaded = false;
				}
			});

			$(self.nodeClass).on('grid.rendered', function() {
				// get gridHeight again after render.
				gridHeight = $(self.wrapperClass).height();

				if ( ! rendered ) {
					self._getTotalPages();
				}
				
				$(self.nodeClass).find('.pp-content-grid-loader').hide();

				setTimeout(function() {
					self._gridLayoutMatchHeightSimple();
					if ( $(self.wrapperClass).data( 'isotope' ) ) {
						$(self.wrapperClass).isotope('layout');
					}
				}, 500);

				// set loaded flag.
				if ( currentPage >= self.totalPages ) {
					loaded = true;
				} else {
					loaded = false;
				}

				rendered = true;
			});

			// Reset pagination index on filter.
			$(self.nodeClass).on('grid.filter.complete', function() {
				if ( $(self.nodeClass).find( '.pp-content-grid-pagination' ).length > 0 ) {
					self._getTotalPages();
					self.currentPage = currentPage = 1;
				}
			});

			self.scrollPaginationIninitiated = true;
		},

		_reLayout: function() {
			var self = this;
			var wrap = $(this.wrapperClass);

			var refreshLayout = function() {
				wrap.imagesLoaded(function() {
					if ( $('body').hasClass('fl-builder-active') ) {
						return;
					}
					if ( ! self.masonry ) {
						self._gridLayoutMatchHeight();
					}

					setTimeout(function() {
						if ( wrap.data( 'isotope' ) ) {
							wrap.isotope('destroy');
						}
						//wrap.isotope( self.filterData );
						self.layoutRefreshed = true;
						self._gridLayout();
						self._initPagination();
					}, 500);
				});
			};

			// Search and Filter.
			$(document).on( 'sf:ajaxfinish', '.searchandfilter', refreshLayout );

			// FacetWP.
			$(document).on( 'facetwp-loaded', function() {
				if ( 'undefined' !== typeof FWPBB && 'undefined' !== typeof FWPBB.modules ) {
					FWPBB.modules[ self.settings.id ] = self.settings;
				}

				refreshLayout();
			} );

			// WP Grid Builder.
			$(window).on( 'wpgb.loaded', function() {
				if ( 'undefined' === typeof WP_Grid_Builder ) {
					return;
				}
				if ( 'undefined' === typeof WP_Grid_Builder['on'] ) {
					return;
				}

				WP_Grid_Builder.on( 'init', function( wpgb ) {
					if ( wpgb.facets && 'undefined' !== typeof wpgb.facets['on'] ) {
						wpgb.facets.on( 'appended', refreshLayout );
					}
				} );
			} );
		}
	};

})(jQuery);


var ppcg_fsem3t8u0knl = '';

;(function($) {
	var left_arrow_svg  = '<span aria-hidden="true"><svg role="img" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 192 512"><path fill="currentColor" d="M25.1 247.5l117.8-116c4.7-4.7 12.3-4.7 17 0l7.1 7.1c4.7 4.7 4.7 12.3 0 17L64.7 256l102.2 100.4c4.7 4.7 4.7 12.3 0 17l-7.1 7.1c-4.7 4.7-12.3 4.7-17 0L25 264.5c-4.6-4.7-4.6-12.3.1-17z"></path></svg></span><span class="sr-only">Previous</span>';
	var right_arrow_svg = '<span aria-hidden="true"><svg role="img" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 192 512"><path fill="currentColor" d="M166.9 264.5l-117.8 116c-4.7 4.7-12.3 4.7-17 0l-7.1-7.1c-4.7-4.7-4.7-12.3 0-17L127.3 256 25.1 155.6c-4.7-4.7-4.7-12.3 0-17l7.1-7.1c4.7-4.7 12.3-4.7 17 0l117.8 116c4.6 4.7 4.6 12.3-.1 17z"></path></svg></span><span class="sr-only">Next</span>';

	var PPContentGridOptions = {
		id: 'fsem3t8u0knl',
		layout: 'carousel',
		style: 'custom',
		scrollTo: true,
		fields: {"layout":"carousel","post_grid_style_select":"custom","alternate_content":"no","custom_layout":{"html":"<div class=\"pp-content-grid-post-text\">\n <div class=\"pp-content-grid-post-excerpt\">\n    \t[wpbb post:content]<br >\n    <\/div>\n\n    <p class=\"pp-content-grid-post-title\">[wpbb post:title][wpbb-if post:custom_field key='position'], [wpbb post:custom_field key='position'][\/wpbb-if]\n<\/p>\n\n    \n\n\n   \n   \n\n<\/div>\n","css":".pp-content-grid-post-text {\n    padding: 20px;\n    font-family: \"DM Serif Text\",serif;\n    font-size: 24px;\n    font-style: italic;\n    text-align: center;\n}\n.pp-content-grid-post-title {\n    font-size: 24px;\n\tline-height: 26px;\n\tmargin: 0;\n\tpadding: 0;\n\tfont-family: \"Red Hat Display\", sans-serif;\n\n\tfont-style: normal;\n}\n\n@media only screen and (max-width: 600px) {\n .pp-content-grid-post-text {\n    padding: 20px;\n    font-family: \"DM Serif Text\",serif;\n    font-size: 18px;\n    font-style: italic;\n    text-align: center;\n}\n}","connections":{"html":""}},"total_post":"all","total_posts_count":"20","posts_per_page":"30","facetwp":"disable","post_slider_arrow_font_size":"36","post_content_width":"60","post_content_width_large":"","post_content_width_medium":"","post_content_width_responsive":"","pagination":"none","load_more_text":"Load More","no_results_message":"Sorry, we couldn't find any posts. Please try a different search.","show_search":"yes","pagination_nofollow":"no","title_tag":"h3","bt_condition_type":"none","bt_display_value1":"","bt_display_relationship":"equals","bt_display_value2":"","bt_cookie_name":"","bt_cookie_relationship":"equals","bt_cookie_value":"","bt_url_text":"","bt_url_type":"exists","bt_url_value":"","bt_time_schedule":"none","bt_start_date":"","bt_start_time":{"hours":"01","minutes":"00","day_period":"am"},"bt_end_date":"","bt_end_time":{"hours":"01","minutes":"00","day_period":"am"},"bt_timezone":"UTC","bt_location_type":"countrycode","bt_location_condition":"equals","bt_location_name":"","bt_display_action":"hide","bt_default_visibility":"show","visibility_logic":"[]","bt_default_module":"","export":"","import":"","type":"pp-content-grid","bb_css_code":"","bb_js_code":"","responsive_display_filtered":true,"connections":{"arrow_color":"","arrow_hover_color":"","arrow_bg_color":"","arrow_bg_hover_color":"","arrow_border":"","arrow_border_hover_color":"","post_slider_dot_bg_color":"","post_slider_dot_bg_hover":"","custom_content":"","more_link_text":"","post_grid_filters_default":"","all_filter_label":"","fallback_image_custom":"","post_bg_color":"","post_bg_color_hover":"","post_border_group":"","post_border_group_large":"","post_border_group_medium":"","post_border_group_responsive":"","post_title_divider_color":"","post_category_bg_color":"","post_category_text_color":"","post_title_overlay_color":"","post_date_day_bg_color":"","post_date_day_text_color":"","post_date_month_bg_color":"","post_date_month_text_color":"","post_date_bg_color":"","post_date_text_color":"","product_rating_color":"","product_price_color":"","button_bg_color":"","button_bg_hover_color":"","button_text_color":"","button_text_hover_color":"","button_border_group":"","button_border_group_large":"","button_border_group_medium":"","button_border_group_responsive":"","button_border_hover_color":"","filter_bg_color":"","filter_bg_color_active":"","filter_text_color":"","filter_text_color_active":"","filter_border_group":"","filter_border_group_large":"","filter_border_group_medium":"","filter_border_group_responsive":"","filter_border_hover_color":"","filter_toggle_bg":"","filter_toggle_color":"","filter_toggle_border_group":"","pagination_bg_color":"","pagination_bg_color_hover":"","pagination_color":"","pagination_color_hover":"","pagination_border_group":"","pagination_border_group_large":"","pagination_border_group_medium":"","pagination_border_group_responsive":"","title_typography":"","title_typography_large":"","title_typography_medium":"","title_typography_responsive":"","title_font_color":"","title_font_hover_color":"","content_typography":"","content_typography_large":"","content_typography_medium":"","content_typography_responsive":"","content_font_color":"","content_font_hover_color":"","meta_typography":"","meta_typography_large":"","meta_typography_medium":"","meta_typography_responsive":"","post_meta_font_color":"","post_meta_font_hover_color":"","post_meta_bg_color":"","event_date_color":"","event_venue_color":"","event_cost_color":"","button_typography":"","button_typography_large":"","button_typography_medium":"","button_typography_responsive":"","filter_typography":"","filter_typography_large":"","filter_typography_medium":"","filter_typography_responsive":"","bt_display_value1":"","bt_display_value2":"","bt_cookie_name":"","bt_cookie_value":"","bt_url_text":"","bt_url_value":"","bt_start_date":"","bt_start_time":"","bt_end_date":"","bt_end_time":"","bt_location_name":""},"pods_source_type":"pods_relation","pods_source_relation":"post_author","pods_source_settings_relation":"","data_source":"custom_query","data_source_acf_relational_type":"relationship","data_source_acf_relational_key":"","data_source_acf_order":"DESC","data_source_acf_order_by":"post__in","data_source_acf_order_by_meta_key":"","post_type":["testimonial"],"order_by":"date","order_by_meta_key":"","order":"DESC","offset":"0","exclude_current_post":"no","posts_post_matching":"1","posts_post":"","tax_post_category_matching":"1","tax_post_category":"","tax_post_post_tag_matching":"1","tax_post_post_tag":"","posts_page_matching":"1","posts_page":"","posts_testimonial_matching":"1","posts_testimonial":"","users_matching":"1","users":"","custom_field_relation":"AND","custom_field":[""],"show_title":"yes","show_content":"yes","content_type":"excerpt","custom_content":"","content_length":"300","more_link_type":"none","more_link_text":"Read More","link_target_new":"no","post_grid_filters_display":"no","post_grid_filters_type":"dynamic","post_grid_filters_terms":"","post_grid_filters_archive_terms":"","post_grid_filters_default":"","all_filter_label":"All","post_grid_filters_order_by":"name","post_grid_filters_order_by_meta_key":"","post_grid_filters_order":"ASC","show_image":"no","image_thumb_size":"large","image_thumb_crop":"","fallback_image":"default","fallback_image_custom":"","show_author":"no","show_date":"no","show_categories":"no","meta_separator":" | ","terms_separator":" \/ ","as_values_posts_post":"","as_values_tax_post_category":"","as_values_tax_post_post_tag":"","as_values_posts_page":"","as_values_posts_testimonial":"","as_values_users":"","post_grid_filters":"","fallback_image_custom_src":"","post_taxonomies":"","posts_case_study_matching":"1","posts_case_study":"","tax_case_study_industry_matching":"1","tax_case_study_industry":"","posts_portfolio_matching":"1","posts_portfolio":"","tax_portfolio_industry_matching":"1","tax_portfolio_industry":"","as_values_posts_case_study":"","as_values_tax_case_study_industry":"","as_values_posts_portfolio":"","as_values_tax_portfolio_industry":"","posts_wpdmpro_matching":"1","posts_wpdmpro":"","tax_wpdmpro_wpdmcategory_matching":"1","tax_wpdmpro_wpdmcategory":"","tax_wpdmpro_wpdmtag_matching":"1","tax_wpdmpro_wpdmtag":"","posts_news_matching":"1","posts_news":"","tax_news_news_type_matching":"1","tax_news_news_type":"","as_values_posts_wpdmpro":"","as_values_tax_wpdmpro_wpdmcategory":"","as_values_tax_wpdmpro_wpdmtag":"","as_values_posts_news":"","as_values_tax_news_news_type":""},
		pagination: 'none',
		postSpacing: '20',
		postColumns: {"xl":1,"lg":1,"md":1,"sm":1},
		breakpoints: {"large":1200,"medium":992,"small":768},
		matchHeight: 'yes',
		filters: false,
		defaultFilter: '',
					filterTax: '',
				filterType: 'dynamic',
							carousel: {
				items: 1,
				responsive: {
					0: {
						items: 1,
						margin: 0,
					},
					1201: {
						items: 1,
						margin: 0,
					},
					993: {
						items: 1,
						margin: 0,
					},
					769: {
						items: 1,
						margin: 0,
					},
				},
							slideBy: 1,
										dots: false,
										autoplay: true,
				autoplayTimeout: 8000,
				autoplaySpeed: 1000,
				autoplayHoverPause: false,
							navSpeed: 1000,
				dotsSpeed: 1000,
				nav: true,
				lazyLoad: false,
				navText : [left_arrow_svg, right_arrow_svg],
				navContainer: '.fl-node-fsem3t8u0knl .pp-carousel-nav',
				navElement:'button type="button"',
				responsiveRefreshRate: 200,
				responsiveBaseWidth: window,
				loop: true,
				center: false,
				autoHeight: false,
				URLhashListener: true			}
				};

	PPContentGridOptions = $.extend( {}, PPContentGridOptions, bb_powerpack.conditionals );

	$(function() {
		ppcg_fsem3t8u0knl = new PPContentGrid( PPContentGridOptions );
	});
	
	// expandable row fix.
	var state = 0;
	$(document).on('pp_expandable_row_toggle', function(e, selector) {
		if ( selector.is('.pp-er-open') && state === 0 && selector.parent().find( '.pp-content-post-grid' ).length > 0 ) {
			if ( 'undefined' !== typeof $.fn.isotope && selector.parent().find('.pp-content-post-grid').data('isotope') ) {
				selector.parent().find('.pp-content-post-grid').isotope('layout');
			}
			state = 1;
		}
	});

	// Tabs and Content Grid fix
	$(document).on('pp-tabs-switched', function(e, selector) {
		if ( selector.find('.pp-content-post-grid').length > 0 ) {
			var postsWrapper = selector.find('.pp-content-post-grid');
			ppcg_fsem3t8u0knl._gridLayoutMatchHeight();
			if ( 'undefined' !== typeof $.fn.isotope && postsWrapper.data('isotope') ) {
				setTimeout(function() {
					postsWrapper.isotope('layout');
				}, 500);
			}
		}
	});

})(jQuery);

/* Start Layout Custom JS */

/* End Layout Custom JS */

