/*!
 * Bowser - a browser detector
 * https://github.com/ded/bowser
 * MIT License | (c) Dustin Diaz 2015
 */

!function (name, definition) {
  if (typeof module != 'undefined' && module.exports) module.exports = definition()
  else if (typeof define == 'function' && define.amd) define(name, definition)
  else this[name] = definition()
}('bowser', function () {
  /**
    * See useragents.js for examples of navigator.userAgent
    */

  var t = true

  function detect(ua) {

    function getFirstMatch(regex) {
      var match = ua.match(regex);
      return (match && match.length > 1 && match[1]) || '';
    }

    function getSecondMatch(regex) {
      var match = ua.match(regex);
      return (match && match.length > 1 && match[2]) || '';
    }

    var iosdevice = getFirstMatch(/(ipod|iphone|ipad)/i).toLowerCase()
      , likeAndroid = /like android/i.test(ua)
      , android = !likeAndroid && /android/i.test(ua)
      , nexusMobile = /nexus\s*[0-6]\s*/i.test(ua)
      , nexusTablet = !nexusMobile && /nexus\s*[0-9]+/i.test(ua)
      , chromeos = /CrOS/.test(ua)
      , silk = /silk/i.test(ua)
      , sailfish = /sailfish/i.test(ua)
      , tizen = /tizen/i.test(ua)
      , webos = /(web|hpw)os/i.test(ua)
      , windowsphone = /windows phone/i.test(ua)
      , windows = !windowsphone && /windows/i.test(ua)
      , mac = !iosdevice && !silk && /macintosh/i.test(ua)
      , linux = !android && !sailfish && !tizen && !webos && /linux/i.test(ua)
      , edgeVersion = getFirstMatch(/edge\/(\d+(\.\d+)?)/i)
      , versionIdentifier = getFirstMatch(/version\/(\d+(\.\d+)?)/i)
      , tablet = /tablet/i.test(ua)
      , mobile = !tablet && /[^-]mobi/i.test(ua)
      , xbox = /xbox/i.test(ua)
      , result

    if (/opera|opr|opios/i.test(ua)) {
      result = {
        name: 'Opera'
      , opera: t
      , version: versionIdentifier || getFirstMatch(/(?:opera|opr|opios)[\s\/](\d+(\.\d+)?)/i)
      }
    }
    else if (/coast/i.test(ua)) {
      result = {
        name: 'Opera Coast'
        , coast: t
        , version: versionIdentifier || getFirstMatch(/(?:coast)[\s\/](\d+(\.\d+)?)/i)
      }
    }
    else if (/yabrowser/i.test(ua)) {
      result = {
        name: 'Yandex Browser'
      , yandexbrowser: t
      , version: versionIdentifier || getFirstMatch(/(?:yabrowser)[\s\/](\d+(\.\d+)?)/i)
      }
    }
    else if (/ucbrowser/i.test(ua)) {
      result = {
          name: 'UC Browser'
        , ucbrowser: t
        , version: getFirstMatch(/(?:ucbrowser)[\s\/](\d+(?:\.\d+)+)/i)
      }
    }
    else if (/mxios/i.test(ua)) {
      result = {
        name: 'Maxthon'
        , maxthon: t
        , version: getFirstMatch(/(?:mxios)[\s\/](\d+(?:\.\d+)+)/i)
      }
    }
    else if (/epiphany/i.test(ua)) {
      result = {
        name: 'Epiphany'
        , epiphany: t
        , version: getFirstMatch(/(?:epiphany)[\s\/](\d+(?:\.\d+)+)/i)
      }
    }
    else if (/puffin/i.test(ua)) {
      result = {
        name: 'Puffin'
        , puffin: t
        , version: getFirstMatch(/(?:puffin)[\s\/](\d+(?:\.\d+)?)/i)
      }
    }
    else if (/sleipnir/i.test(ua)) {
      result = {
        name: 'Sleipnir'
        , sleipnir: t
        , version: getFirstMatch(/(?:sleipnir)[\s\/](\d+(?:\.\d+)+)/i)
      }
    }
    else if (/k-meleon/i.test(ua)) {
      result = {
        name: 'K-Meleon'
        , kMeleon: t
        , version: getFirstMatch(/(?:k-meleon)[\s\/](\d+(?:\.\d+)+)/i)
      }
    }
    else if (windowsphone) {
      result = {
        name: 'Windows Phone'
      , windowsphone: t
      }
      if (edgeVersion) {
        result.msedge = t
        result.version = edgeVersion
      }
      else {
        result.msie = t
        result.version = getFirstMatch(/iemobile\/(\d+(\.\d+)?)/i)
      }
    }
    else if (/msie|trident/i.test(ua)) {
      result = {
        name: 'Internet Explorer'
      , msie: t
      , version: getFirstMatch(/(?:msie |rv:)(\d+(\.\d+)?)/i)
      }
    } else if (chromeos) {
      result = {
        name: 'Chrome'
      , chromeos: t
      , chromeBook: t
      , chrome: t
      , version: getFirstMatch(/(?:chrome|crios|crmo)\/(\d+(\.\d+)?)/i)
      }
    } else if (/chrome.+? edge/i.test(ua)) {
      result = {
        name: 'Microsoft Edge'
      , msedge: t
      , version: edgeVersion
      }
    }
    else if (/vivaldi/i.test(ua)) {
      result = {
        name: 'Vivaldi'
        , vivaldi: t
        , version: getFirstMatch(/vivaldi\/(\d+(\.\d+)?)/i) || versionIdentifier
      }
    }
    else if (sailfish) {
      result = {
        name: 'Sailfish'
      , sailfish: t
      , version: getFirstMatch(/sailfish\s?browser\/(\d+(\.\d+)?)/i)
      }
    }
    else if (/seamonkey\//i.test(ua)) {
      result = {
        name: 'SeaMonkey'
      , seamonkey: t
      , version: getFirstMatch(/seamonkey\/(\d+(\.\d+)?)/i)
      }
    }
    else if (/firefox|iceweasel|fxios/i.test(ua)) {
      result = {
        name: 'Firefox'
      , firefox: t
      , version: getFirstMatch(/(?:firefox|iceweasel|fxios)[ \/](\d+(\.\d+)?)/i)
      }
      if (/\((mobile|tablet);[^\)]*rv:[\d\.]+\)/i.test(ua)) {
        result.firefoxos = t
      }
    }
    else if (silk) {
      result =  {
        name: 'Amazon Silk'
      , silk: t
      , version : getFirstMatch(/silk\/(\d+(\.\d+)?)/i)
      }
    }
    else if (/phantom/i.test(ua)) {
      result = {
        name: 'PhantomJS'
      , phantom: t
      , version: getFirstMatch(/phantomjs\/(\d+(\.\d+)?)/i)
      }
    }
    else if (/slimerjs/i.test(ua)) {
      result = {
        name: 'SlimerJS'
        , slimer: t
        , version: getFirstMatch(/slimerjs\/(\d+(\.\d+)?)/i)
      }
    }
    else if (/blackberry|\bbb\d+/i.test(ua) || /rim\stablet/i.test(ua)) {
      result = {
        name: 'BlackBerry'
      , blackberry: t
      , version: versionIdentifier || getFirstMatch(/blackberry[\d]+\/(\d+(\.\d+)?)/i)
      }
    }
    else if (webos) {
      result = {
        name: 'WebOS'
      , webos: t
      , version: versionIdentifier || getFirstMatch(/w(?:eb)?osbrowser\/(\d+(\.\d+)?)/i)
      };
      if( /touchpad\//i.test(ua) ){
        result.touchpad = t;
      }
    }
    else if (/bada/i.test(ua)) {
      result = {
        name: 'Bada'
      , bada: t
      , version: getFirstMatch(/dolfin\/(\d+(\.\d+)?)/i)
      };
    }
    else if (tizen) {
      result = {
        name: 'Tizen'
      , tizen: t
      , version: getFirstMatch(/(?:tizen\s?)?browser\/(\d+(\.\d+)?)/i) || versionIdentifier
      };
    }
    else if (/qupzilla/i.test(ua)) {
      result = {
        name: 'QupZilla'
        , qupzilla: t
        , version: getFirstMatch(/(?:qupzilla)[\s\/](\d+(?:\.\d+)+)/i) || versionIdentifier
      }
    }
    else if (/chromium/i.test(ua)) {
      result = {
        name: 'Chromium'
        , chromium: t
        , version: getFirstMatch(/(?:chromium)[\s\/](\d+(?:\.\d+)?)/i) || versionIdentifier
      }
    }
    else if (/chrome|crios|crmo/i.test(ua)) {
      result = {
        name: 'Chrome'
        , chrome: t
        , version: getFirstMatch(/(?:chrome|crios|crmo)\/(\d+(\.\d+)?)/i)
      }
    }
    else if (android) {
      result = {
        name: 'Android'
        , version: versionIdentifier
      }
    }
    else if (/safari|applewebkit/i.test(ua)) {
      result = {
        name: 'Safari'
      , safari: t
      }
      if (versionIdentifier) {
        result.version = versionIdentifier
      }
    }
    else if (iosdevice) {
      result = {
        name : iosdevice == 'iphone' ? 'iPhone' : iosdevice == 'ipad' ? 'iPad' : 'iPod'
      }
      // WTF: version is not part of user agent in web apps
      if (versionIdentifier) {
        result.version = versionIdentifier
      }
    }
    else if(/googlebot/i.test(ua)) {
      result = {
        name: 'Googlebot'
      , googlebot: t
      , version: getFirstMatch(/googlebot\/(\d+(\.\d+))/i) || versionIdentifier
      }
    }
    else {
      result = {
        name: getFirstMatch(/^(.*)\/(.*) /),
        version: getSecondMatch(/^(.*)\/(.*) /)
     };
   }

    // set webkit or gecko flag for browsers based on these engines
    if (!result.msedge && /(apple)?webkit/i.test(ua)) {
      if (/(apple)?webkit\/537\.36/i.test(ua)) {
        result.name = result.name || "Blink"
        result.blink = t
      } else {
        result.name = result.name || "Webkit"
        result.webkit = t
      }
      if (!result.version && versionIdentifier) {
        result.version = versionIdentifier
      }
    } else if (!result.opera && /gecko\//i.test(ua)) {
      result.name = result.name || "Gecko"
      result.gecko = t
      result.version = result.version || getFirstMatch(/gecko\/(\d+(\.\d+)?)/i)
    }

    // set OS flags for platforms that have multiple browsers
    if (!result.msedge && (android || result.silk)) {
      result.android = t
    } else if (iosdevice) {
      result[iosdevice] = t
      result.ios = t
    } else if (mac) {
      result.mac = t
    } else if (xbox) {
      result.xbox = t
    } else if (windows) {
      result.windows = t
    } else if (linux) {
      result.linux = t
    }

    // OS version extraction
    var osVersion = '';
    if (result.windowsphone) {
      osVersion = getFirstMatch(/windows phone (?:os)?\s?(\d+(\.\d+)*)/i);
    } else if (iosdevice) {
      osVersion = getFirstMatch(/os (\d+([_\s]\d+)*) like mac os x/i);
      osVersion = osVersion.replace(/[_\s]/g, '.');
    } else if (android) {
      osVersion = getFirstMatch(/android[ \/-](\d+(\.\d+)*)/i);
    } else if (result.webos) {
      osVersion = getFirstMatch(/(?:web|hpw)os\/(\d+(\.\d+)*)/i);
    } else if (result.blackberry) {
      osVersion = getFirstMatch(/rim\stablet\sos\s(\d+(\.\d+)*)/i);
    } else if (result.bada) {
      osVersion = getFirstMatch(/bada\/(\d+(\.\d+)*)/i);
    } else if (result.tizen) {
      osVersion = getFirstMatch(/tizen[\/\s](\d+(\.\d+)*)/i);
    }
    if (osVersion) {
      result.osversion = osVersion;
    }

    // device type extraction
    var osMajorVersion = osVersion.split('.')[0];
    if (
         tablet
      || nexusTablet
      || iosdevice == 'ipad'
      || (android && (osMajorVersion == 3 || (osMajorVersion >= 4 && !mobile)))
      || result.silk
    ) {
      result.tablet = t
    } else if (
         mobile
      || iosdevice == 'iphone'
      || iosdevice == 'ipod'
      || android
      || nexusMobile
      || result.blackberry
      || result.webos
      || result.bada
    ) {
      result.mobile = t
    }

    // Graded Browser Support
    // http://developer.yahoo.com/yui/articles/gbs
    if (result.msedge ||
        (result.msie && result.version >= 10) ||
        (result.yandexbrowser && result.version >= 15) ||
		    (result.vivaldi && result.version >= 1.0) ||
        (result.chrome && result.version >= 20) ||
        (result.firefox && result.version >= 20.0) ||
        (result.safari && result.version >= 6) ||
        (result.opera && result.version >= 10.0) ||
        (result.ios && result.osversion && result.osversion.split(".")[0] >= 6) ||
        (result.blackberry && result.version >= 10.1)
        || (result.chromium && result.version >= 20)
        ) {
      result.a = t;
    }
    else if ((result.msie && result.version < 10) ||
        (result.chrome && result.version < 20) ||
        (result.firefox && result.version < 20.0) ||
        (result.safari && result.version < 6) ||
        (result.opera && result.version < 10.0) ||
        (result.ios && result.osversion && result.osversion.split(".")[0] < 6)
        || (result.chromium && result.version < 20)
        ) {
      result.c = t
    } else result.x = t

    return result
  }

  var bowser = detect(typeof navigator !== 'undefined' ? navigator.userAgent : '')

  bowser.test = function (browserList) {
    for (var i = 0; i < browserList.length; ++i) {
      var browserItem = browserList[i];
      if (typeof browserItem=== 'string') {
        if (browserItem in bowser) {
          return true;
        }
      }
    }
    return false;
  }

  /**
   * Get version precisions count
   *
   * @example
   *   getVersionPrecision("1.10.3") // 3
   *
   * @param  {string} version
   * @return {number}
   */
  function getVersionPrecision(version) {
    return version.split(".").length;
  }

  /**
   * Array::map polyfill
   *
   * @param  {Array} arr
   * @param  {Function} iterator
   * @return {Array}
   */
  function map(arr, iterator) {
    var result = [], i;
    if (Array.prototype.map) {
      return Array.prototype.map.call(arr, iterator);
    }
    for (i = 0; i < arr.length; i++) {
      result.push(iterator(arr[i]));
    }
    return result;
  }

  /**
   * Calculate browser version weight
   *
   * @example
   *   compareVersions(['1.10.2.1',  '1.8.2.1.90'])    // 1
   *   compareVersions(['1.010.2.1', '1.09.2.1.90']);  // 1
   *   compareVersions(['1.10.2.1',  '1.10.2.1']);     // 0
   *   compareVersions(['1.10.2.1',  '1.0800.2']);     // -1
   *
   * @param  {Array<String>} versions versions to compare
   * @return {Number} comparison result
   */
  function compareVersions(versions) {
    // 1) get common precision for both versions, for example for "10.0" and "9" it should be 2
    var precision = Math.max(getVersionPrecision(versions[0]), getVersionPrecision(versions[1]));
    var chunks = map(versions, function (version) {
      var delta = precision - getVersionPrecision(version);

      // 2) "9" -> "9.0" (for precision = 2)
      version = version + new Array(delta + 1).join(".0");

      // 3) "9.0" -> ["000000000"", "000000009"]
      return map(version.split("."), function (chunk) {
        return new Array(20 - chunk.length).join("0") + chunk;
      }).reverse();
    });

    // iterate in reverse order by reversed chunks array
    while (--precision >= 0) {
      // 4) compare: "000000009" > "000000010" = false (but "9" > "10" = true)
      if (chunks[0][precision] > chunks[1][precision]) {
        return 1;
      }
      else if (chunks[0][precision] === chunks[1][precision]) {
        if (precision === 0) {
          // all version chunks are same
          return 0;
        }
      }
      else {
        return -1;
      }
    }
  }

  /**
   * Check if browser is unsupported
   *
   * @example
   *   bowser.isUnsupportedBrowser({
   *     msie: "10",
   *     firefox: "23",
   *     chrome: "29",
   *     safari: "5.1",
   *     opera: "16",
   *     phantom: "534"
   *   });
   *
   * @param  {Object}  minVersions map of minimal version to browser
   * @param  {Boolean} [strictMode = false] flag to return false if browser wasn't found in map
   * @param  {String}  [ua] user agent string
   * @return {Boolean}
   */
  function isUnsupportedBrowser(minVersions, strictMode, ua) {
    var _bowser = bowser;

    // make strictMode param optional with ua param usage
    if (typeof strictMode === 'string') {
      ua = strictMode;
      strictMode = void(0);
    }

    if (strictMode === void(0)) {
      strictMode = false;
    }
    if (ua) {
      _bowser = detect(ua);
    }

    var version = "" + _bowser.version;
    for (var browser in minVersions) {
      if (minVersions.hasOwnProperty(browser)) {
        if (_bowser[browser]) {
          // browser version and min supported version.
          return compareVersions([version, minVersions[browser]]) < 0;
        }
      }
    }

    return strictMode; // not found
  }

  /**
   * Check if browser is supported
   *
   * @param  {Object} minVersions map of minimal version to browser
   * @param  {Boolean} [strictMode = false] flag to return false if browser wasn't found in map
   * @param  {String}  [ua] user agent string
   * @return {Boolean}
   */
  function check(minVersions, strictMode, ua) {
    return !isUnsupportedBrowser(minVersions, strictMode, ua);
  }

  bowser.isUnsupportedBrowser = isUnsupportedBrowser;
  bowser.compareVersions = compareVersions;
  bowser.check = check;

  /*
   * Set our detect method to the main bowser object so we can
   * reuse it to test other user agents.
   * This is needed to implement future tests.
   */
  bowser._detect = detect;

  return bowser
});

(function($){
  UABBTrigger = {

      /**
       * Trigger a hook.
       *
       * @since 1.1.0.3
       * @method triggerHook
       * @param {String} hook The hook to trigger.
       * @param {Array} args An array of args to pass to the hook.
       */
      triggerHook: function( hook, args )
      {
        $( 'body' ).trigger( 'uabb-trigger.' + hook, args );
      },
    
      /**
       * Add a hook.
       *
       * @since 1.1.0.3
       * @method addHook
       * @param {String} hook The hook to add.
       * @param {Function} callback A function to call when the hook is triggered.
       */
      addHook: function( hook, callback )
      {
        $( 'body' ).on( 'uabb-trigger.' + hook, callback );
      },
    
      /**
       * Remove a hook.
       *
       * @since 1.1.0.3
       * @method removeHook
       * @param {String} hook The hook to remove.
       * @param {Function} callback The callback function to remove.
       */
      removeHook: function( hook, callback )
      {
        $( 'body' ).off( 'uabb-trigger.' + hook, callback );
      },
  };
})(jQuery);

jQuery(document).ready(function( $ ) {

    if( typeof bowser !== 'undefined' && bowser !== null ) {

      var uabb_browser   = bowser.name,
          uabb_browser_v = bowser.version,
          uabb_browser_class = uabb_browser.replace(/\s+/g, '-').toLowerCase(),
          uabb_browser_v_class = uabb_browser_class + parseInt( uabb_browser_v );
      
      $('html').addClass(uabb_browser_class).addClass(uabb_browser_v_class);
      
    }

    $('.uabb-row-separator').parents('html').css('overflow-x', 'hidden');
});

;(function($) {

	PPModalBox = function(settings) {
		this.id 			= settings.id;
		this.settings 		= settings;
		this.type			= settings.type;
		this.cookieKey		= 'pp_modal_' + this.id;
		this.cookieTime 	= settings.display_after;
		this.cookieValue 	= settings.cookie_value ? settings.cookie_value : this.cookieTime;
		this.triggerType 	= settings.trigger_type;
		this.layout			= settings.layout;
		this.wrap 			= $('#modal-' + this.id);
		this.container 		= this.wrap.find('.pp-modal-container');
		this.element 		= this.wrap.find('.pp-modal');
		this.isPreviewing 	= settings.previewing;
		this.isVisible		= settings.visible;
		this.eventClose		= false;

		this.initCookieApi();
		this.init();
	};

	PPModalBox.prototype = {
		id			: '',
		settings	: {},
		type		: '',
		cookieKey	: '',
		cookieTime	: 0,
		triggerType	: '',
		layout		: '',
		wrap		: '',
		element		: '',
		prevFocusedEl: '',
		isActive	: false,
		isPreviewing: false,
		isVisible	: false,
		eventClose  : false,

		init: function()
		{
			if ( parseInt( this.cookieTime ) === 0 || this.cookieTime < 0 || this.cookieTime === '' ) {
				this.removeCookie();
			}
			if ( ( 'exit_intent' === this.triggerType || 'auto' === this.triggerType ) && this.getCookie() && ! this.isPreviewing ) {
				return;
			}
			if ( ! this.isPreviewing && 'undefined' !== typeof this.isVisible && ! this.isVisible ) {
				return;
			}
			if ( this.isActive ) {
				return;
			}

			this.setResponsive();
			this.bindEvents();
			this.show();
		},

		setResponsive: function()
		{
			if ( window.innerWidth <= this.settings.breakpoint ) {
                this.element.removeClass('layout-standard').addClass('layout-fullscreen');
            }
            if ( window.innerWidth < this.element.width() ) {
				this.element.css('width', window.innerWidth + 'px');
            }
		},

		bindEvents: function()
		{
			var self = this;

			$( this.element ).on( 'beforeload', this.beforeLoad.bind( this ) );

			$(document).on( 'keyup', function(e) {
				// close modal box on Esc key press.
                if ( self.settings.esc_exit && 27 == e.which && self.isActive && $('form[data-type="pp-modal-box"]').length === 0 ) {
					self.eventClose = true;
                    self.hide();
                }
			}).on( 'keydown', function(e) {
				// trap focus inside the modal element.
				if ( (e.key === 'Tab' || e.keyCode === 9) && self.isActive ) {
					if ( e.shiftKey && $(document.activeElement).is( self.element ) ) {
						e.preventDefault();
					}
					if ( $(document.activeElement).closest( self.element ).length === 0 ) {
						self.element.trigger('focus');
						e.preventDefault();
					}
				}
			} );

			// close modal box by clicking on outside of modal box element in document.
			$(document).on( 'click', function(e) {
                if ( self.settings.click_exit && $(e.target).parents('.pp-modal').length === 0 && self.isActive && ! self.isPreviewing && ! self.element.is(e.target) && self.element.has(e.target).length === 0 && ! $(e.target).hasClass('modal-' + self.id) && $(e.target).parents('.modal-' + self.id).length === 0 && e.which ) {
					self.eventClose = true;
                    self.hide();
                }
			});
			
			// close modal box by clicking on the close button.
            $(self.wrap).find('.pp-modal-close, .pp-modal-close-custom').on('keypress click', function(e) {
				e.preventDefault();
				e.stopPropagation();
				if (e.which == 1 || e.which == 13 || e.which == 32 || e.which == undefined) {
					self.eventClose = true;
					self.hide();
				}
			});

			// close the modal box on clicking on inner links start with hash.
			$('body').on( 'click', '#modal-' + self.id + ' .pp-modal-content a[href*="#"]:not([href="#"])', function() {
				if ( $('.fl-node-' + self.id).hasClass('anchor-click-no-event') ) {
					return;
				}
				self.eventClose = true;
                self.hide();
			} );

			$(window).on( 'resize', this.setResponsive.bind( this ) );
			$(window).on( 'resize', this.setPosition.bind( this ) );
		},

		setPosition: function()
		{
			if ( 'fullscreen' !== this.layout ) {
                if ( typeof this.settings.height === 'undefined' ) {

                    this.wrap.addClass('pp-modal-height-auto');
                    var modalHeight = this.element.outerHeight();
                    this.wrap.removeClass('pp-modal-height-auto');

                    if ( 'photo' === this.type ) {
                        this.element.find( '.pp-modal-content-inner img' ).css( 'max-width', '100%' );
                    }

                    var topPos = ( window.innerHeight - modalHeight ) / 2;
                    if ( topPos < 0 ) {
                        topPos = 0;
                    }
                    this.element.css( 'top', topPos + 'px' );
                } else {
                    var topPos = ( window.innerHeight - this.settings.height ) / 2;
					if ( topPos < 0 ) {
                        topPos = 0;
                    }
                    this.element.css( 'top', topPos + 'px' );
                }
			}
		},

		beforeLoad: function() {
			if ( this.settings.clickedElement ) {
				var clickedElement = this.settings.clickedElement;
				var postId = clickedElement.attr( 'data-pp-modal-post' ) || clickedElement.parents( '.pp-content-post' ).attr( 'data-id' );
				var self = this;

				if ( 'undefined' === typeof postId || '' === postId ) {
					return;
				}

				self.element.find( '.pp-modal-content-inner' ).html('<div style="text-align: center;"><img src="' + self.settings.loaderImg + '" /></div>');
				self.setPosition();

				if ( 'html' === self.settings.type && '' !== self.settings.content ) {
					$.ajax({
						url: bb_powerpack.getAjaxUrl(),
						type: 'post',
						data: {
							action: 'pp_modal_dynamic_content',
							content: self.settings.content,
							postId: postId
						},
						success: function( response ) {
							if ( ! response.success ) {
								return;
							}

							self.element.find( '.pp-modal-content-inner' ).html( response.data );
							self.setPosition();
						}
					});
				}
			}
		},

		show: function()
		{
			if ( this.element.length === 0 ) {
				return;
			}

			this.setPosition();

			var self = this;

			if ( 'photo' === this.type ) {
				this.element.find( '.pp-modal-content-inner img' ).css( 'max-width', '100%' );
			}

			this.prevFocusedEl = $( document.activeElement );

			//this.element.find( '.pp-modal-content' ).scrollTop(0);
			
			setTimeout( function() {
				self.element.trigger('beforeload');

				if ( ! self.isPreviewing ) {
					setTimeout(function() {
						self.element.attr( 'tabindex', '0' ).trigger( 'focus' );
					}, 100);
				}

				$('html').addClass( 'pp-modal-active-' + self.id );

				self.wrap.addClass( 'pp-modal-active' );

				self.container
					.removeClass( self.settings.animation_load + ' animated' )
					.addClass( 'modal-visible' )
					.addClass( self.settings.animation_load + ' animated' );

				if ( ! $('body').hasClass('wp-admin') ) {
					self.container.one( 'webkitAnimationEnd mozAnimationEnd MSAnimationEnd oanimationend animationend', function() {
						self.eventClose = false;
						$(this).removeClass( self.settings.animation_load + ' animated' );
						self.setup();
					} );
				} else {
					self.setup();
				}

				self.isActive = true;
				
                if ( 'exit_intent' === self.triggerType || 'auto' === self.triggerType ){
                    if ( ! self.isPreviewing ) {
                        self.setCookie();
                    }
				}
				
                self.restruct();
				self.setPosition();

				self.element.trigger('afterload');
				$(document).trigger( 'pp_modal_box_rendered', [self.element] );
				
            }, self.settings.auto_load ? parseFloat(self.settings.delay) * 1000 : 0);
		},

		setup: function()
		{
			if ( this.element.find('.pp-modal-iframe').length > 0 ) {
				var original_src = this.element.find('.pp-modal-iframe').attr('src');
				var src = this.element.find('.pp-modal-iframe').data('url');
				if ( original_src === undefined || original_src === '' ) {
					this.element.find('.pp-modal-iframe').attr( 'src', src );
				}
			}

			var iframeAndSource = this.element.find('iframe, source');
			if ( iframeAndSource.length > 0 && iframeAndSource.closest( '.fl-module' ).length === 0 ) {
				var src = '';
				var m_src = iframeAndSource.attr('src');
				
				if ( m_src === undefined || m_src === '' ) {
					src = iframeAndSource.data('url');
				} else {
					src = iframeAndSource.attr('src');
				}

				if ( src ) {
					if ( ( src.search('youtube') !== -1 || src.search('vimeo') !== -1) && src.search('autoplay=1') == -1 ) {
						if ( typeof src.split('?')[1] === 'string' ) {
							src = src + '&autoplay=1&rel=0';
						} else {
							src = src + '?autoplay=1&rel=0';
						}
					}
					iframeAndSource.attr('src', src);
				}
			}

			if ( this.element.find('video').length ) {
				this.element.find('video')[0].play();
			}
		},

		reset: function()
		{
            var iframeAndSource = this.element.find('iframe, source');
			if ( iframeAndSource.length > 0 && iframeAndSource.closest( '.fl-module' ).length === 0 ) {
				var src = iframeAndSource.attr('src');
				if ( '' !== src ) {
					iframeAndSource.attr('data-url', src).attr('src', '');
				}
            }
			if ( this.element.find('video').length > 0 ) {
				this.element.find('video')[0].pause();
			}
			if ( this.element.find('mux-player').length > 0 ) {
				this.element.find('mux-player')[0].pause();
			}

			this.element.attr( 'tabindex', '-1' );
			var self = this;
			
			setTimeout(function() {
				if ( self.settings.clickedElement ) {
					var $clickedElement = self.settings.clickedElement;
					if ( $clickedElement.closest( '.modal-' + self.id ).length ) {
						$clickedElement.closest( '.modal-' + self.id ).trigger( 'focus' );
					} else if ( $clickedElement.closest( '#trigger-' + self.id ).length ) {
						$clickedElement.closest( '#trigger-' + self.id ).trigger( 'focus' );
					} else if ( '' !== self.settings.customTrigger ) {
						if ( $clickedElement.is( self.settings.customTrigger ) ) {
							$clickedElement.trigger( 'focus' );
						} else if ( $clickedElement.closest( self.settings.customTrigger ).length ) {
							$clickedElement.closest( self.settings.customTrigger ).trigger( 'focus' );
						}
					}
				}
			}, 100);
        },

		restruct: function()
		{
			var mH = 0, hH = 0, cH = 0, eq = 0;
			var self = this;

            setTimeout( function() {
                if ( self.isActive ) {
                    if ( 'fullscreen' === self.layout ) {
                        var marginTop 		= parseInt( self.element.css('margin-top') );
                        var marginBottom 	= parseInt( self.element.css('margin-bottom') );
                        var modalHeight 	= $(window).height() - (marginTop + marginBottom);
						
						self.element.css( 'height', modalHeight + 'px' );
                    }
                    eq = 6;
                    mH = self.element.outerHeight(); // Modal height.
                    hH = self.element.find('.pp-modal-header').outerHeight(); // Header height.

                    if ( self.settings.auto_height && 'fullscreen' !== self.layout ) {
                        return;
					}
					
					var cP = parseInt( self.element.find('.pp-modal-content').css('padding') ); // Content padding.
					
					self.element.find('.pp-modal-content').css( 'height', mH - (hH + eq) + 'px' );
					
                    if ( ! self.settings.auto_height && self.element.find('.pp-modal-header').length === 0) {
                        self.element.find('.pp-modal-content').css('height', mH + 'px');
                    }
				   
					// Adjust iframe height.
                    if ( 'url' === self.type && self.element.find( '.pp-modal-video-embed' ).length === 0 ) {
                        self.element.find('.pp-modal-iframe').css('height', self.element.find('.pp-modal-content-inner').outerHeight() + 'px');
                    }
                    if ( 'video' === self.type ) {
                        self.element.find('iframe').css({'height':'100%', 'width':'100%'});
                    }
                }
            }, self.settings.auto_load ? parseFloat(self.settings.delay) * 1000 : 1);
		},

		hide: function()
		{
			var self = this;

			this.element.trigger('beforeclose');

            this.container
                .removeClass( self.settings.animation_exit + ' animated' )
				.addClass( self.settings.animation_exit + ' animated' );
				
			if ( ! $('body').hasClass('wp-admin') ) {
				this.container.one( 'webkitAnimationEnd mozAnimationEnd MSAnimationEnd oanimationend animationend', function() {
					if ( self.eventClose ) {
						self.close();
					}
				});
			} else {
				self.close();
			}
				
            if ( window.location.hash ) {
				var hashVal = window.location.hash.replace('/', '');
                if ( '#modal-' + self.id === hashVal ) {
                    var scrollTop = self.settings.scrollTop || $(window).scrollTop();
                    window.location.href = window.location.href.split('#')[0] + '#';
                    $(window).scrollTop(scrollTop);
                }
			}

			if ( ! this.isPreviewing ) {
				this.element.attr( 'tabindex', '-1' ).trigger( 'blur' );
				if ( this.prevFocusedEl && this.prevFocusedEl.length > 0 ) {
					this.prevFocusedEl.trigger( 'focus' );
				}
			}
			
			this.element.trigger('afterclose');
			$(document).trigger( 'pp_modal_box_after_close', [this.element, this] );
		},

		close: function()
		{
			this.container.removeClass( this.settings.animation_exit + ' animated' ).removeClass('modal-visible');
			this.container.find('.pp-modal-content').removeAttr('style');
			this.wrap.removeClass( 'pp-modal-active' );

			$('html').removeClass( 'pp-modal-active-' + this.id );

			this.isActive = false;
			this.eventClose = false;
			this.reset();
		},

		initCookieApi: function() {
			if ( 'undefined' === typeof $.cookie ) {
				/*!
				 * jQuery Cookie Plugin v1.4.1
				 * https://github.com/carhartl/jquery-cookie
				 *
				 * Copyright 2013 Klaus Hartl
				 * Released under the MIT license
				*/
				!function(a){"function"==typeof define&&define.amd?define(["jquery"],a):a("object"==typeof exports?require("jquery"):jQuery)}(function(a){function b(a){return h.raw?a:encodeURIComponent(a)}function c(a){return h.raw?a:decodeURIComponent(a)}function d(a){return b(h.json?JSON.stringify(a):String(a))}function e(a){0===a.indexOf('"')&&(a=a.slice(1,-1).replace(/\\"/g,'"').replace(/\\\\/g,"\\"));try{return a=decodeURIComponent(a.replace(g," ")),h.json?JSON.parse(a):a}catch(b){}}function f(b,c){var d=h.raw?b:e(b);return a.isFunction(c)?c(d):d}var g=/\+/g,h=a.cookie=function(e,g,i){if(void 0!==g&&!a.isFunction(g)){if(i=a.extend({},h.defaults,i),"number"==typeof i.expires){var j=i.expires,k=i.expires=new Date;k.setTime(+k+864e5*j)}return document.cookie=[b(e),"=",d(g),i.expires?"; expires="+i.expires.toUTCString():"",i.path?"; path="+i.path:"",i.domain?"; domain="+i.domain:"",i.secure?"; secure":""].join("")}for(var l=e?void 0:{},m=document.cookie?document.cookie.split("; "):[],n=0,o=m.length;o>n;n++){var p=m[n].split("="),q=c(p.shift()),r=p.join("=");if(e&&e===q){l=f(r,g);break}e||void 0===(r=f(r))||(l[q]=r)}return l};h.defaults={},a.removeCookie=function(b,c){return void 0===a.cookie(b)?!1:(a.cookie(b,"",a.extend({},c,{expires:-1})),!a.cookie(b))}});
			}
		},

		setCookie: function()
		{
			if ( parseInt( this.cookieTime ) > 0 ) {
				return $.cookie( this.cookieKey, this.cookieValue, {expires: this.cookieTime, path: '/'} );
			} else {
				this.removeCookie();
			}
		},

		getCookie: function()
		{
			// Reset cookie when module settings change.
			if ( this.cookieValue != $.cookie( this.cookieKey ) ) {
				this.removeCookie();
			}
			return $.cookie( this.cookieKey );
		},

		removeCookie: function()
		{
			$.cookie( this.cookieKey, 0, {expires: 0, path: '/'} );
		}
	};
})(jQuery);

var pp_modal_xe3qrmzu4igk = false;
var pp_modal_xe3qrmzu4igk_config = false;

;(function($) {

    pp_modal_xe3qrmzu4igk_config = {
        id: 'xe3qrmzu4igk',
        type: 'templates',
		trigger_type: 'auto',
        auto_load: true,
        exit_intent: false,
                display_after: 1,
                		load_on_scroll: 0,
        		        delay: 1,
        animation_load: 'fadeIn',
        animation_exit: 'fadeOut',
		overlay_animation: true,
        esc_exit: true,
        click_exit: true,
        layout: 'standard',
        auto_height: true,
                width: 600,
        breakpoint: 0,
        		loaderImg: 'https://mangiero.sumy-starter.us/wp-content/plugins/bbpowerpack/assets/images/ajax-loader.gif',
        previewing: false    };

	
    $(document).on('click', function(e) {
        if ( e && e.target.tagName === 'A' && e.target.href.indexOf('#modal-xe3qrmzu4igk') !== -1 ) {
            pp_modal_xe3qrmzu4igk_config['scrollTop'] = $(window).scrollTop();
        }
    });

        $(function() {
        $('#modal-xe3qrmzu4igk').appendTo(document.body);

        var tabHash     = window.location.hash ? window.location.hash.replace('/', '') : false;
        var modalId     = window.location.hash.split('#modal-')[1];

        // If the URL contains a hash beginning with modal, trigger that modal box.
        if ( tabHash && tabHash.indexOf('modal-') >= 0 ) {
            if ( modalId === 'xe3qrmzu4igk' ) {
                pp_modal_xe3qrmzu4igk = new PPModalBox(pp_modal_xe3qrmzu4igk_config);
            }
        }

        $(window).on('hashchange', function() {
            var tabHash     = window.location.hash;
            var modalId     = window.location.hash.split('#modal-')[1];

            // If the URL contains a hash beginning with modal, trigger that modal box.
            if ( tabHash && tabHash.indexOf('modal-') >= 0 ) {
                if ( modalId === 'xe3qrmzu4igk' ) {
					if ( pp_modal_xe3qrmzu4igk instanceof PPModalBox ) {
						pp_modal_xe3qrmzu4igk.show();
					} else {
                    	pp_modal_xe3qrmzu4igk = new PPModalBox(pp_modal_xe3qrmzu4igk_config);
					}
                }
            }
        });

    });

		
								pp_modal_xe3qrmzu4igk = new PPModalBox(pp_modal_xe3qrmzu4igk_config);
    				    
	// Bind the click event to any element with the class.
	pp_modal_xe3qrmzu4igk_config.customTrigger = '';
    $(document).on('click keydown', '.modal-xe3qrmzu4igk', function(e) {
		var valid = (e.which == 1 || e.which == 13 || e.which == 32 || e.which == undefined);
		if ( ! valid ) {
			return;
		}
        e.preventDefault();
		if ( pp_modal_xe3qrmzu4igk instanceof PPModalBox ) {
			pp_modal_xe3qrmzu4igk.settings.clickedElement = $( e.target );
			pp_modal_xe3qrmzu4igk.show();
        } else {
			pp_modal_xe3qrmzu4igk_config.clickedElement = $( e.target );
        	pp_modal_xe3qrmzu4igk = new PPModalBox(pp_modal_xe3qrmzu4igk_config);
        }
    });

    
})(jQuery);

var pp_modal_8wj1bp7ziqsg = false;
var pp_modal_8wj1bp7ziqsg_config = false;

;(function($) {

    pp_modal_8wj1bp7ziqsg_config = {
        id: '8wj1bp7ziqsg',
        type: 'templates',
		trigger_type: 'auto',
        auto_load: true,
        exit_intent: false,
                display_after: 1,
                		load_on_scroll: 0,
        		        delay: 1,
        animation_load: 'fadeIn',
        animation_exit: 'fadeOut',
		overlay_animation: true,
        esc_exit: true,
        click_exit: true,
        layout: 'standard',
        auto_height: true,
                width: 600,
        breakpoint: 0,
        		loaderImg: 'https://mangiero.sumy-starter.us/wp-content/plugins/bbpowerpack/assets/images/ajax-loader.gif',
        previewing: false    };

	
    $(document).on('click', function(e) {
        if ( e && e.target.tagName === 'A' && e.target.href.indexOf('#modal-8wj1bp7ziqsg') !== -1 ) {
            pp_modal_8wj1bp7ziqsg_config['scrollTop'] = $(window).scrollTop();
        }
    });

        $(function() {
        $('#modal-8wj1bp7ziqsg').appendTo(document.body);

        var tabHash     = window.location.hash ? window.location.hash.replace('/', '') : false;
        var modalId     = window.location.hash.split('#modal-')[1];

        // If the URL contains a hash beginning with modal, trigger that modal box.
        if ( tabHash && tabHash.indexOf('modal-') >= 0 ) {
            if ( modalId === '8wj1bp7ziqsg' ) {
                pp_modal_8wj1bp7ziqsg = new PPModalBox(pp_modal_8wj1bp7ziqsg_config);
            }
        }

        $(window).on('hashchange', function() {
            var tabHash     = window.location.hash;
            var modalId     = window.location.hash.split('#modal-')[1];

            // If the URL contains a hash beginning with modal, trigger that modal box.
            if ( tabHash && tabHash.indexOf('modal-') >= 0 ) {
                if ( modalId === '8wj1bp7ziqsg' ) {
					if ( pp_modal_8wj1bp7ziqsg instanceof PPModalBox ) {
						pp_modal_8wj1bp7ziqsg.show();
					} else {
                    	pp_modal_8wj1bp7ziqsg = new PPModalBox(pp_modal_8wj1bp7ziqsg_config);
					}
                }
            }
        });

    });

		
								pp_modal_8wj1bp7ziqsg = new PPModalBox(pp_modal_8wj1bp7ziqsg_config);
    				    
	// Bind the click event to any element with the class.
	pp_modal_8wj1bp7ziqsg_config.customTrigger = '';
    $(document).on('click keydown', '.modal-8wj1bp7ziqsg', function(e) {
		var valid = (e.which == 1 || e.which == 13 || e.which == 32 || e.which == undefined);
		if ( ! valid ) {
			return;
		}
        e.preventDefault();
		if ( pp_modal_8wj1bp7ziqsg instanceof PPModalBox ) {
			pp_modal_8wj1bp7ziqsg.settings.clickedElement = $( e.target );
			pp_modal_8wj1bp7ziqsg.show();
        } else {
			pp_modal_8wj1bp7ziqsg_config.clickedElement = $( e.target );
        	pp_modal_8wj1bp7ziqsg = new PPModalBox(pp_modal_8wj1bp7ziqsg_config);
        }
    });

    
})(jQuery);

/* Start Layout Custom JS */

/* End Layout Custom JS */

